/**
 * Copyright (c) 2016 itemis AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Alexander Nyßen (itemis AG) - initial API and implementation
 */
package org.eclipse.gef4.dot.internal;

import com.google.common.base.Objects;
import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javafx.collections.ObservableList;
import javafx.collections.ObservableMap;
import org.eclipse.gef4.common.attributes.IAttributeStore;
import org.eclipse.gef4.dot.internal.DotAttributes;
import org.eclipse.gef4.dot.internal.DotFileUtils;
import org.eclipse.gef4.dot.internal.parser.dot.EdgeOp;
import org.eclipse.gef4.graph.Edge;
import org.eclipse.gef4.graph.Graph;
import org.eclipse.gef4.graph.Node;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A serializer that creates a Graphviz DOT string or file from a {@link Graph} with {@link DotAttributes}.
 * 
 * @author anyssen
 */
@SuppressWarnings("all")
public class DotExport {
  public String export(final Graph graph) {
    String _xblockexpression = null;
    {
      ObservableMap<String, Object> _attributes = graph.getAttributes();
      boolean _containsKey = _attributes.containsKey(DotAttributes._TYPE__G);
      boolean _not = (!_containsKey);
      if (_not) {
        throw new IllegalArgumentException(
          (((("The " + DotAttributes._TYPE__G) + " attribute has to be set on the input graph ") + graph) + "."));
      }
      ObservableList<Node> _nodes = graph.getNodes();
      final Function1<Node, Boolean> _function = new Function1<Node, Boolean>() {
        public Boolean apply(final Node it) {
          ObservableMap<String, Object> _attributes = it.getAttributes();
          Object _get = _attributes.get(DotAttributes._NAME__GNE);
          return Boolean.valueOf(Objects.equal(_get, null));
        }
      };
      boolean _exists = IterableExtensions.<Node>exists(_nodes, _function);
      if (_exists) {
        throw new IllegalArgumentException(
          (((("The " + DotAttributes._NAME__GNE) + " attribute has to be set for all nodes of the input graph ") + graph) + "."));
      }
      _xblockexpression = this.print(graph);
    }
    return _xblockexpression;
  }
  
  public File export(final Graph graph, final String pathname) {
    String _export = this.export(graph);
    File _file = new File(pathname);
    return DotFileUtils.write(_export, _file);
  }
  
  private String print(final Graph graph) {
    StringConcatenation _builder = new StringConcatenation();
    ObservableMap<String, Object> _attributes = graph.getAttributes();
    Object _get = _attributes.get(DotAttributes._TYPE__G);
    _builder.append(_get, "");
    _builder.append(" ");
    {
      boolean _hasName = this.hasName(graph);
      if (_hasName) {
        String _name = this.name(graph);
        _builder.append(_name, "");
        _builder.append(" ");
      }
    }
    _builder.append("{");
    _builder.newLineIfNotEmpty();
    {
      boolean _hasNonMetaAttributes = this.hasNonMetaAttributes(graph);
      if (_hasNonMetaAttributes) {
        _builder.append("\t");
        String _printNonMetaAttributes = this.printNonMetaAttributes(graph, ";");
        _builder.append(_printNonMetaAttributes, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    ObservableList<Node> _nodes = graph.getNodes();
    final Function1<Node, String> _function = new Function1<Node, String>() {
      public String apply(final Node it) {
        ObservableMap<String, Object> _attributes = it.getAttributes();
        Object _get = _attributes.get(DotAttributes._NAME__GNE);
        return ((String) _get);
      }
    };
    List<Node> _sortBy = IterableExtensions.<Node, String>sortBy(_nodes, _function);
    final Function1<Node, String> _function_1 = new Function1<Node, String>() {
      public String apply(final Node it) {
        return DotExport.this.print(it);
      }
    };
    List<String> _map = ListExtensions.<Node, String>map(_sortBy, _function_1);
    String _join = IterableExtensions.join(_map, "; ");
    _builder.append(_join, "\t");
    _builder.newLineIfNotEmpty();
    {
      ObservableList<Edge> _edges = graph.getEdges();
      final Function1<Edge, String> _function_2 = new Function1<Edge, String>() {
        public String apply(final Edge it) {
          ObservableMap<String, Object> _attributes = it.getAttributes();
          Object _get = _attributes.get(DotAttributes._NAME__GNE);
          return ((String) _get);
        }
      };
      List<Edge> _sortBy_1 = IterableExtensions.<Edge, String>sortBy(_edges, _function_2);
      for(final Edge edge : _sortBy_1) {
        _builder.append("\t");
        String _name_1 = this.name(edge);
        _builder.append(_name_1, "\t");
        {
          boolean _hasNonMetaAttributes_1 = this.hasNonMetaAttributes(edge);
          if (_hasNonMetaAttributes_1) {
            _builder.append(" [");
            String _printNonMetaAttributes_1 = this.printNonMetaAttributes(edge, ",");
            _builder.append(_printNonMetaAttributes_1, "\t");
            _builder.append("]");
          }
        }
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    _builder.newLine();
    return _builder.toString();
  }
  
  private boolean isMetaAttribute(final String key) {
    return key.startsWith("_");
  }
  
  private boolean isDirected(final Graph graph) {
    ObservableMap<String, Object> _attributes = graph.getAttributes();
    Object _get = _attributes.get(DotAttributes._TYPE__G);
    return DotAttributes._TYPE__G__DIGRAPH.equals(_get);
  }
  
  private String print(final Node node) {
    String _name = this.name(node);
    String _xifexpression = null;
    boolean _hasNonMetaAttributes = this.hasNonMetaAttributes(node);
    if (_hasNonMetaAttributes) {
      String _printNonMetaAttributes = this.printNonMetaAttributes(node, ",");
      String _plus = (" [" + _printNonMetaAttributes);
      _xifexpression = (_plus + "]");
    } else {
      _xifexpression = "";
    }
    return (_name + _xifexpression);
  }
  
  private boolean hasName(final IAttributeStore it) {
    ObservableMap<String, Object> _attributes = it.getAttributes();
    Object _get = _attributes.get(DotAttributes._NAME__GNE);
    return (!Objects.equal(_get, null));
  }
  
  private String _name(final IAttributeStore store) {
    ObservableMap<String, Object> _attributes = store.getAttributes();
    Object _get = _attributes.get(DotAttributes._NAME__GNE);
    return ((String) _get);
  }
  
  private String _name(final Edge edge) {
    Node _source = edge.getSource();
    String _name = this.name(_source);
    String _xifexpression = null;
    Graph _graph = edge.getGraph();
    boolean _isDirected = this.isDirected(_graph);
    if (_isDirected) {
      _xifexpression = EdgeOp.DIRECTED.getLiteral();
    } else {
      _xifexpression = EdgeOp.UNDIRECTED.getLiteral();
    }
    String _plus = (_name + _xifexpression);
    Node _target = edge.getTarget();
    String _name_1 = this.name(_target);
    return (_plus + _name_1);
  }
  
  private boolean hasNonMetaAttributes(final IAttributeStore store) {
    ObservableMap<String, Object> _attributes = store.getAttributes();
    Set<String> _keySet = _attributes.keySet();
    final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
      public Boolean apply(final String it) {
        boolean _isMetaAttribute = DotExport.this.isMetaAttribute(it);
        return Boolean.valueOf((!_isMetaAttribute));
      }
    };
    return IterableExtensions.<String>exists(_keySet, _function);
  }
  
  private String printNonMetaAttributes(final IAttributeStore store, final String separator) {
    ObservableMap<String, Object> _attributes = store.getAttributes();
    Set<Map.Entry<String, Object>> _entrySet = _attributes.entrySet();
    final Function1<Map.Entry<String, Object>, Boolean> _function = new Function1<Map.Entry<String, Object>, Boolean>() {
      public Boolean apply(final Map.Entry<String, Object> it) {
        String _key = it.getKey();
        boolean _isMetaAttribute = DotExport.this.isMetaAttribute(_key);
        return Boolean.valueOf((!_isMetaAttribute));
      }
    };
    Iterable<Map.Entry<String, Object>> _filter = IterableExtensions.<Map.Entry<String, Object>>filter(_entrySet, _function);
    final Function1<Map.Entry<String, Object>, String> _function_1 = new Function1<Map.Entry<String, Object>, String>() {
      public String apply(final Map.Entry<String, Object> it) {
        String _key = it.getKey();
        String _plus = (_key + "=\"");
        Object _value = it.getValue();
        String _plus_1 = (_plus + _value);
        return (_plus_1 + "\"");
      }
    };
    Iterable<String> _map = IterableExtensions.<Map.Entry<String, Object>, String>map(_filter, _function_1);
    List<String> _sort = IterableExtensions.<String>sort(_map);
    return IterableExtensions.join(_sort, (separator + " "));
  }
  
  private String name(final IAttributeStore edge) {
    if (edge instanceof Edge) {
      return _name((Edge)edge);
    } else if (edge != null) {
      return _name(edge);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(edge).toString());
    }
  }
}
