/*******************************************************************************
 * Copyright (c) 2015 itemis AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Alexander Nyßen (itemis AG) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gef4.common.adapt;

import java.util.Map;

import org.eclipse.gef4.common.adapt.inject.InjectAdapters;

import com.google.common.reflect.TypeToken;

import javafx.beans.property.ReadOnlyMapProperty;
import javafx.collections.ObservableMap;

/**
 * An {@link AdapterStore} is a basic {@link IAdaptable} implementation that can
 * be used standalone.
 * 
 * @author anyssen
 */
public class AdapterStore implements IAdaptable {

	private AdaptableSupport<AdapterStore> ads = new AdaptableSupport<>(this);

	/**
	 * Creates a new {@link AdapterStore} with no initial adapters.
	 */
	public AdapterStore() {
	}

	/**
	 * Creates a new AdapterStore with the single given initial adapter, using
	 * the 'default' role.
	 * 
	 * @param <T>
	 *            The adapter type.
	 * @param adapter
	 *            The adapter to be registered.
	 * 
	 */
	public <T> AdapterStore(T adapter) {
		setAdapter(adapter, AdapterKey.DEFAULT_ROLE);
	}

	/**
	 * Creates a new AdapterStore with the single given initial adapter, using
	 * the 'default' role.
	 * 
	 * @param <T>
	 *            The adapter type.
	 * @param adapterType
	 *            The runtime type of the adapter to be registered.
	 * @param adapter
	 *            The adapter to be registered.
	 * 
	 */
	public <T> AdapterStore(TypeToken<T> adapterType, T adapter) {
		setAdapter(adapterType, adapter, AdapterKey.DEFAULT_ROLE);
	}

	/**
	 * Creates a new AdapterStore with the single given initial adapter.
	 * 
	 * @param <T>
	 *            The adapter type.
	 * @param adapterType
	 *            The runtime type of the adapter to be registered.
	 * @param adapter
	 *            The adapter to be registered.
	 * @param role
	 *            The role under which the adapter is to be registered.
	 */
	public <T> AdapterStore(TypeToken<T> adapterType, T adapter, String role) {
		setAdapter(adapterType, adapter, role);
	}

	@Override
	public <T> T getAdapter(AdapterKey<T> key) {
		return ads.getAdapter(key);
	}

	@Override
	public <T> T getAdapter(Class<T> key) {
		return ads.getAdapter(key);
	}

	@Override
	public <T> T getAdapter(TypeToken<T> key) {
		return ads.getAdapter(key);
	}

	@Override
	public <T> Map<AdapterKey<? extends T>, T> getAdapters(
			Class<? super T> key) {
		return ads.getAdapters(key);
	}

	@Override
	public <T> Map<AdapterKey<? extends T>, T> getAdapters(
			TypeToken<? super T> key) {
		return ads.getAdapters(key);
	}

	@Override
	public <T> void unsetAdapter(T adapter) {
		ads.unsetAdapter(adapter);
	}

	@Override
	public <T> void setAdapter(T adapter) {
		ads.setAdapter(adapter);
	}

	@Override
	public <T> void setAdapter(TypeToken<T> adapterType, T adapter) {
		ads.setAdapter(adapterType, adapter);
	}

	@Override
	public <T> void setAdapter(T adapter, String role) {
		ads.setAdapter(adapter, role);
	}

	@InjectAdapters
	@Override
	public <T> void setAdapter(TypeToken<T> adapterType, T adapter,
			String role) {
		ads.setAdapter(adapterType, adapter, role);
	}

	/**
	 * Removes all registered adapters from this {@link AdapterStore}.
	 */
	public void clear() {
		for (Object adapter : ads.getAdapters().values()) {
			ads.unsetAdapter(adapter);
		}
	}

	@Override
	public ReadOnlyMapProperty<AdapterKey<?>, Object> adaptersProperty() {
		return ads.adaptersProperty();
	}

	@Override
	public ObservableMap<AdapterKey<?>, Object> getAdapters() {
		return ads.getAdapters();
	}

}