/**
 * Copyright (c) 2016 itemis AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Alexander Nyßen (itemis AG) - initial API and implementation
 */
package org.eclipse.gef4.dot.internal;

import com.google.inject.Injector;
import java.io.File;
import java.io.StringReader;
import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef4.dot.internal.DotFileUtils;
import org.eclipse.gef4.dot.internal.DotInterpreter;
import org.eclipse.gef4.dot.internal.parser.DotStandaloneSetup;
import org.eclipse.gef4.dot.internal.parser.dot.DotAst;
import org.eclipse.gef4.dot.internal.parser.parser.antlr.DotParser;
import org.eclipse.gef4.graph.Graph;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.SyntaxErrorMessage;
import org.eclipse.xtext.parser.IParseResult;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A parser that creates a {@link Graph} with {@link DotAttributes} from a Graphviz DOT string or file.
 * 
 * @author anyssen
 */
@SuppressWarnings("all")
public class DotImport {
  private final static Injector dotInjector = new DotStandaloneSetup().createInjectorAndDoEMFRegistration();
  
  private final static DotParser dotParser = ((DotParser) DotImport.dotInjector.<DotParser>getInstance(DotParser.class));
  
  public Graph importDot(final String dotString) {
    Graph _xblockexpression = null;
    {
      StringReader _stringReader = new StringReader(dotString);
      IParseResult parseResult = DotImport.dotParser.parse(_stringReader);
      boolean _hasSyntaxErrors = parseResult.hasSyntaxErrors();
      if (_hasSyntaxErrors) {
        Iterable<INode> _syntaxErrors = parseResult.getSyntaxErrors();
        final Function1<INode, String> _function = new Function1<INode, String>() {
          public String apply(final INode it) {
            SyntaxErrorMessage _syntaxErrorMessage = it.getSyntaxErrorMessage();
            return _syntaxErrorMessage.getMessage();
          }
        };
        Iterable<String> _map = IterableExtensions.<INode, String>map(_syntaxErrors, _function);
        String _join = IterableExtensions.join(_map, ",");
        String _plus = ("Given DOT string is not valid: " + _join);
        throw new IllegalArgumentException(_plus);
      }
      DotInterpreter _dotInterpreter = new DotInterpreter();
      EObject _rootASTElement = parseResult.getRootASTElement();
      List<Graph> _interpret = _dotInterpreter.interpret(((DotAst) _rootASTElement));
      _xblockexpression = IterableExtensions.<Graph>head(_interpret);
    }
    return _xblockexpression;
  }
  
  public Graph importDot(final File dotFile) {
    String _read = DotFileUtils.read(dotFile);
    return this.importDot(_read);
  }
}
