/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.tasks;

import com.codahale.metrics.DefaultSettableGauge;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.vertx.core.Promise;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.cassandra.sidecar.cluster.InstancesMetadata;
import org.apache.cassandra.sidecar.cluster.instance.InstanceMetadata;
import org.apache.cassandra.sidecar.common.server.utils.DurationSpec;
import org.apache.cassandra.sidecar.config.CdcConfiguration;
import org.apache.cassandra.sidecar.config.ServiceConfiguration;
import org.apache.cassandra.sidecar.db.SystemViewsDatabaseAccessor;
import org.apache.cassandra.sidecar.exceptions.SchemaUnavailableException;
import org.apache.cassandra.sidecar.metrics.DeltaGauge;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.metrics.server.CdcMetrics;
import org.apache.cassandra.sidecar.tasks.PeriodicTask;
import org.apache.cassandra.sidecar.tasks.ScheduleDecision;
import org.apache.cassandra.sidecar.utils.CdcUtil;
import org.apache.cassandra.sidecar.utils.FileUtils;
import org.apache.cassandra.sidecar.utils.TimeProvider;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class CdcRawDirectorySpaceCleaner
implements PeriodicTask {
    private static final Logger LOGGER = LoggerFactory.getLogger(CdcRawDirectorySpaceCleaner.class);
    public static final String CDC_DIR_NAME = "cdc_raw";
    private final TimeProvider timeProvider;
    private final SystemViewsDatabaseAccessor systemViewsDatabaseAccessor;
    private final CdcConfiguration cdcConfiguration;
    private final InstancesMetadata instancesMetadata;
    private final CdcMetrics cdcMetrics;
    @Nullable
    private Long maxUsageBytes = null;
    private Long maxUsageLastReadNanos = null;
    private Map<CdcRawSegmentFile, Long> priorCdcFiles = new HashMap<CdcRawSegmentFile, Long>();

    @Inject
    public CdcRawDirectorySpaceCleaner(TimeProvider timeProvider, SystemViewsDatabaseAccessor systemViewsDatabaseAccessor, ServiceConfiguration serviceConfiguration, InstancesMetadata instancesMetadata, SidecarMetrics metrics) {
        this.timeProvider = timeProvider;
        this.systemViewsDatabaseAccessor = systemViewsDatabaseAccessor;
        this.cdcConfiguration = serviceConfiguration.cdcConfiguration();
        this.instancesMetadata = instancesMetadata;
        this.cdcMetrics = metrics.server().cdc();
    }

    @Override
    public DurationSpec delay() {
        return this.cdcConfiguration.cdcRawDirectorySpaceCleanerFrequency();
    }

    @Override
    public void execute(Promise<Void> promise) {
        try {
            this.routineCleanUp();
            promise.tryComplete();
        }
        catch (Throwable t) {
            LOGGER.warn("Failed to perform routine clean-up of cdc_raw directory", t);
            ((DeltaGauge)this.cdcMetrics.cdcRawCleanerFailed.metric).update(1L);
            promise.fail(t);
        }
    }

    protected boolean shouldRefreshCachedMaxUsage() {
        return this.maxUsageLastReadNanos == null || this.timeProvider.nanoTime() - this.maxUsageLastReadNanos >= TimeUnit.MILLISECONDS.toNanos(this.cdcConfiguration.cacheMaxUsage().toMillis());
    }

    protected long maxUsageBytes() {
        if (!this.shouldRefreshCachedMaxUsage()) {
            return Objects.requireNonNull(this.maxUsageBytes, "maxUsageBytes cannot be null if maxUsageLastReadNanos is non-null");
        }
        try {
            Long newValue = this.systemViewsDatabaseAccessor.cdcTotalSpaceBytesSetting();
            if (newValue != null) {
                if (!newValue.equals(this.maxUsageBytes)) {
                    LOGGER.info("Change in cdc_total_space from system_views.settings prev={} latest={}", (Object)this.maxUsageBytes, (Object)newValue);
                    this.maxUsageBytes = newValue;
                }
                this.maxUsageLastReadNanos = this.timeProvider.nanoTime();
                return this.maxUsageBytes;
            }
        }
        catch (SchemaUnavailableException e) {
            LOGGER.debug("Could not read cdc_total_space from system_views.settings", (Throwable)e);
        }
        catch (Throwable t) {
            LOGGER.error("Error reading cdc_total_space from system_views.settings", t);
        }
        LOGGER.warn("Could not read cdc_total_space from system_views.settings, falling back to props");
        return this.cdcConfiguration.fallbackCdcRawDirectoryMaxSizeBytes();
    }

    @Override
    public ScheduleDecision scheduleDecision() {
        if (this.cdcConfiguration.enableCdcRawDirectoryRoutineCleanUp()) {
            return ScheduleDecision.EXECUTE;
        }
        LOGGER.debug("Skipping CdcRawDirectorySpaceCleaner: feature is disabled");
        return ScheduleDecision.SKIP;
    }

    protected void routineCleanUp() {
        for (InstanceMetadata instanceMetadata : this.instancesMetadata.instances()) {
            String cdcDir = instanceMetadata.cdcDir();
            if (cdcDir != null) {
                try {
                    this.cleanUpCdcRawDirectory(new File(cdcDir));
                }
                catch (Exception e) {
                    LOGGER.warn("Unable to clean up CDC directory {} for instance {}", new Object[]{cdcDir, instanceMetadata, e});
                }
                continue;
            }
            LOGGER.warn("CDC directory is not configured for instance {}. Skipping clean up", (Object)instanceMetadata);
        }
    }

    protected void cleanUpCdcRawDirectory(File cdcRawDirectory) {
        if (!cdcRawDirectory.exists() || !cdcRawDirectory.isDirectory()) {
            LOGGER.debug("Skipping CdcRawDirectorySpaceCleaner: CDC directory does not exist: {}", (Object)cdcRawDirectory);
            return;
        }
        List segmentFiles = Optional.ofNullable(cdcRawDirectory.listFiles(this::validSegmentFilter)).map(files -> Arrays.stream(files).map(CdcRawSegmentFile::new).filter(CdcRawSegmentFile::indexExists).collect(Collectors.toList())).orElseGet(List::of);
        this.publishCdcStats(segmentFiles);
        if (segmentFiles.size() < 2) {
            LOGGER.debug("Skipping cdc data cleaner routine cleanup: No cdc data or only one single cdc segment is found.");
            return;
        }
        long directorySizeBytes = FileUtils.directorySizeBytes(cdcRawDirectory);
        long maxUsageBytes = this.maxUsageBytes();
        long upperLimitBytes = (long)((float)maxUsageBytes * this.cdcConfiguration.cdcRawDirectoryMaxPercentUsage());
        Collections.sort(segmentFiles);
        long nowInMillis = this.timeProvider.currentTimeMillis();
        ((DefaultSettableGauge)this.cdcMetrics.oldestSegmentAge.metric).setValue((Object)((int)TimeUnit.MILLISECONDS.toSeconds(nowInMillis - ((CdcRawSegmentFile)segmentFiles.get(0)).lastModified())));
        LOGGER.debug("Cdc data cleaner directorySizeBytes={} maxedUsageBytes={} upperLimitBytes={}", new Object[]{directorySizeBytes, maxUsageBytes, upperLimitBytes});
        if (directorySizeBytes > upperLimitBytes) {
            long length;
            if (((CdcRawSegmentFile)segmentFiles.get((int)0)).segmentId > ((CdcRawSegmentFile)segmentFiles.get((int)1)).segmentId) {
                LOGGER.error("Cdc segments sorted incorrectly {} before {}", (Object)((CdcRawSegmentFile)segmentFiles.get((int)0)).segmentId, (Object)((CdcRawSegmentFile)segmentFiles.get((int)1)).segmentId);
            }
            long criticalMillis = this.cdcConfiguration.cdcRawDirectoryCriticalBufferWindow().toMillis();
            long lowMillis = this.cdcConfiguration.cdcRawDirectoryLowBufferWindow().toMillis();
            for (int i = 0; i < segmentFiles.size() - 1 && directorySizeBytes > upperLimitBytes; directorySizeBytes -= length, ++i) {
                CdcRawSegmentFile segment = (CdcRawSegmentFile)segmentFiles.get(i);
                long ageMillis = nowInMillis - segment.lastModified();
                if (ageMillis < criticalMillis) {
                    LOGGER.error("Insufficient Cdc buffer size to maintain {}-minute window segment={} maxSize={} ageMinutes={}", new Object[]{TimeUnit.MILLISECONDS.toMinutes(criticalMillis), segment, upperLimitBytes, TimeUnit.MILLISECONDS.toMinutes(ageMillis)});
                    ((DeltaGauge)this.cdcMetrics.criticalCdcRawSpace.metric).update(1L);
                } else if (ageMillis < lowMillis) {
                    LOGGER.warn("Insufficient Cdc buffer size to maintain {}-minute window segment={} maxSize={} ageMinutes={}", new Object[]{TimeUnit.MILLISECONDS.toMinutes(lowMillis), segment, upperLimitBytes, TimeUnit.MILLISECONDS.toMinutes(ageMillis)});
                    ((DeltaGauge)this.cdcMetrics.lowCdcRawSpace.metric).update(1L);
                }
                length = 0L;
                try {
                    length = this.deleteSegment(segment);
                    ((DeltaGauge)this.cdcMetrics.deletedSegment.metric).update(length);
                    continue;
                }
                catch (IOException e) {
                    LOGGER.warn("Failed to delete cdc segment", (Throwable)e);
                }
            }
        }
        try {
            this.cleanupOrphanedIdxFiles(cdcRawDirectory);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to clean up orphaned idx files", (Throwable)e);
        }
    }

    protected boolean validSegmentFilter(File file) {
        return file.isFile() && CdcUtil.isLogFile(file.getName());
    }

    protected long deleteSegment(CdcRawSegmentFile segment) throws IOException {
        long numBytes = segment.length() + segment.indexLength();
        LOGGER.info("Deleting Cdc segment path={} lastModified={} numBytes={}", new Object[]{segment, segment.lastModified(), numBytes});
        Files.deleteIfExists(segment.path());
        Files.deleteIfExists(segment.indexPath());
        return numBytes;
    }

    private void cleanupOrphanedIdxFiles(File cdcDir) throws IOException {
        File[] indexFiles = cdcDir.listFiles(f -> f.isFile() && CdcUtil.isValidIdxFile(f.getName()));
        if (indexFiles == null || indexFiles.length == 0) {
            return;
        }
        File[] cdcSegments = cdcDir.listFiles(f -> f.isFile() && CdcUtil.isLogFile(f.getName()));
        Set<String> cdcFileNames = Set.of();
        if (cdcSegments != null) {
            cdcFileNames = new HashSet(cdcSegments.length);
            for (File f2 : cdcSegments) {
                cdcFileNames.add(f2.getName());
            }
        }
        for (File idxFile : indexFiles) {
            String cdcFileName = CdcUtil.idxToLogFileName(idxFile.getName());
            if (cdcFileNames.contains(cdcFileName)) continue;
            LOGGER.warn("Orphaned Cdc idx file found with no corresponding Cdc segment path={}", (Object)idxFile.toPath());
            ((DeltaGauge)this.cdcMetrics.orphanedIdx.metric).update(1L);
            Files.deleteIfExists(idxFile.toPath());
        }
    }

    private void publishCdcStats(@Nullable List<CdcRawSegmentFile> cdcFiles) {
        HashMap<CdcRawSegmentFile, Long> currentFiles;
        boolean noCdcFiles;
        boolean bl = noCdcFiles = cdcFiles == null || cdcFiles.isEmpty();
        if (noCdcFiles && this.priorCdcFiles.isEmpty()) {
            return;
        }
        long totalCurrentBytes = 0L;
        if (noCdcFiles) {
            currentFiles = new HashMap<CdcRawSegmentFile, Long>();
        } else {
            currentFiles = new HashMap(cdcFiles.size());
            for (CdcRawSegmentFile segment : cdcFiles) {
                if (!segment.exists()) continue;
                long len = segment.length();
                currentFiles.put(segment, len);
                totalCurrentBytes += len;
            }
        }
        if (totalCurrentBytes == 0L && this.priorCdcFiles.isEmpty()) {
            this.priorCdcFiles = currentFiles;
            return;
        }
        Sets.SetView consumedFiles = Sets.difference(this.priorCdcFiles.keySet(), currentFiles.keySet());
        long totalConsumedBytes = consumedFiles.stream().map(this.priorCdcFiles::get).reduce(0L, Long::sum);
        this.priorCdcFiles.clear();
        this.priorCdcFiles = currentFiles;
        ((DeltaGauge)this.cdcMetrics.totalConsumedCdcBytes.metric).update(totalConsumedBytes);
        ((DefaultSettableGauge)this.cdcMetrics.totalCdcSpaceUsed.metric).setValue((Object)totalCurrentBytes);
    }

    protected static class CdcRawSegmentFile
    implements Comparable<CdcRawSegmentFile> {
        private final File file;
        private final File indexFile;
        private final long segmentId;
        private final long len;

        CdcRawSegmentFile(File logFile) {
            this.file = logFile;
            String name = logFile.getName();
            this.segmentId = CdcUtil.parseSegmentId(name);
            this.len = logFile.length();
            this.indexFile = CdcUtil.getIdxFile(logFile);
        }

        public boolean exists() {
            return this.file.exists();
        }

        public boolean indexExists() {
            return this.indexFile.exists();
        }

        public long length() {
            return this.len;
        }

        public long indexLength() {
            return this.indexFile.length();
        }

        public long lastModified() {
            return this.file.lastModified();
        }

        public Path path() {
            return this.file.toPath();
        }

        public Path indexPath() {
            return this.indexFile.toPath();
        }

        @Override
        public int compareTo(@NotNull CdcRawSegmentFile o) {
            return Long.compare(this.segmentId, o.segmentId);
        }

        public int hashCode() {
            return this.file.hashCode();
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || this.getClass() != other.getClass()) {
                return false;
            }
            CdcRawSegmentFile that = (CdcRawSegmentFile)other;
            return this.file.equals(that.file);
        }

        public String toString() {
            return this.file.getAbsolutePath();
        }
    }
}

