// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#include "io/fs/s3_file_reader.h"

#include <aws/core/http/URI.h>
#include <aws/core/utils/Outcome.h>
#include <aws/s3/S3Client.h>
#include <aws/s3/S3Errors.h>
#include <aws/s3/model/GetObjectRequest.h>
#include <aws/s3/model/GetObjectResult.h>
#include <fmt/format.h>
#include <glog/logging.h>

#include <algorithm>
#include <utility>

// IWYU pragma: no_include <opentelemetry/common/threadlocal.h>
#include "common/compiler_util.h" // IWYU pragma: keep
#include "io/fs/s3_common.h"
#include "util/doris_metrics.h"
#include "util/runtime_profile.h"
#include "util/s3_util.h"

namespace doris {
namespace io {
class IOContext;
bvar::Adder<uint64_t> s3_file_reader_read_counter("s3_file_reader", "read_at");
bvar::Adder<uint64_t> s3_file_reader_total("s3_file_reader", "total_num");
bvar::Adder<uint64_t> s3_bytes_read_total("s3_file_reader", "bytes_read");
bvar::Adder<uint64_t> s3_file_being_read("s3_file_reader", "file_being_read");
bvar::Adder<uint64_t> s3_file_reader_too_many_request_counter("s3_file_reader", "too_many_request");
bvar::LatencyRecorder s3_bytes_per_read("s3_file_reader", "bytes_per_read"); // also QPS
bvar::PerSecond<bvar::Adder<uint64_t>> s3_read_througthput("s3_file_reader", "s3_read_throughput",
                                                           &s3_bytes_read_total);
// Although we can get QPS from s3_bytes_per_read, but s3_bytes_per_read only
// record successfull request, and s3_get_request_qps will record all request.
bvar::PerSecond<bvar::Adder<uint64_t>> s3_get_request_qps("s3_file_reader", "s3_get_request",
                                                          &s3_file_reader_read_counter);

S3FileReader::S3FileReader(Path path, size_t file_size, std::string key, std::string bucket,
                           std::shared_ptr<S3FileSystem> fs, RuntimeProfile* profile)
        : _path(std::move(path)),
          _file_size(file_size),
          _fs(std::move(fs)),
          _bucket(std::move(bucket)),
          _key(std::move(key)),
          _profile(profile) {
    DorisMetrics::instance()->s3_file_open_reading->increment(1);
    DorisMetrics::instance()->s3_file_reader_total->increment(1);
    s3_file_reader_total << 1;
    s3_file_being_read << 1;

    Aws::Http::SetCompliantRfc3986Encoding(true);
}

S3FileReader::~S3FileReader() {
    close();
    s3_file_being_read << -1;
}

Status S3FileReader::close() {
    bool expected = false;
    if (_closed.compare_exchange_strong(expected, true, std::memory_order_acq_rel)) {
        DorisMetrics::instance()->s3_file_open_reading->increment(-1);
    }
    return Status::OK();
}

Status S3FileReader::read_at_impl(size_t offset, Slice result, size_t* bytes_read,
                                  const IOContext* /*io_ctx*/) {
    DCHECK(!closed());
    if (offset > _file_size) {
        return Status::IOError("offset exceeds file size(offset: {}, file size: {}, path: {})",
                               offset, _file_size, _path.native());
    }
    size_t bytes_req = result.size;
    char* to = result.data;
    bytes_req = std::min(bytes_req, _file_size - offset);
    if (UNLIKELY(bytes_req == 0)) {
        *bytes_read = 0;
        return Status::OK();
    }

    Aws::S3::Model::GetObjectRequest request;
    request.WithBucket(_bucket).WithKey(_key);
    request.SetRange(fmt::format("bytes={}-{}", offset, offset + bytes_req - 1));
    request.SetResponseStreamFactory(AwsWriteableStreamFactory(to, bytes_req));

    auto client = _fs->get_client();
    if (!client) {
        return Status::InternalError("init s3 client error");
    }
    // auto outcome = client->GetObject(request);
    // if (!outcome.IsSuccess()) {
    //     return Status::IOError(
    //             "failed to read from {}: {}, exception {}, error code {}, request id {}",
    //             _path.native(), outcome.GetError().GetMessage(),
    //             outcome.GetError().GetExceptionName(), outcome.GetError().GetResponseCode(),
    //             outcome.GetError().GetRequestId());
    // }
    // *bytes_read = outcome.GetResult().GetContentLength();
    // if (*bytes_read != bytes_req) {
    //     return Status::IOError("failed to read from {}(bytes read: {}, bytes req: {})",
    //                            _path.native(), *bytes_read, bytes_req);
    // SCOPED_BVAR_LATENCY(s3_bvar::s3_get_latency);

    int retry_count = 0;
    const int base_wait_time = config::s3_read_base_wait_time_ms; // Base wait time in milliseconds
    const int max_wait_time = config::s3_read_max_wait_time_ms; // Maximum wait time in milliseconds
    const int max_retries = config::max_s3_client_retry; // wait 1s, 2s, 4s, 8s for each backoff

    int total_sleep_time = 0;
    while (retry_count <= max_retries) {
        s3_file_reader_read_counter << 1;
        auto outcome = client->GetObject(request);
        if (!outcome.IsSuccess()) {
            auto error = outcome.GetError();
            if (error.GetResponseCode() == Aws::Http::HttpResponseCode::TOO_MANY_REQUESTS) {
                s3_file_reader_too_many_request_counter << 1;
                retry_count++;
                int wait_time = std::min(base_wait_time * (1 << retry_count),
                                         max_wait_time); // Exponential backoff
                std::this_thread::sleep_for(std::chrono::milliseconds(wait_time));
                total_sleep_time += wait_time;
                continue;
            } else {
                // Handle other errors
                return Status::IOError(
                        "failed to read. msg: {}, exception: {}, error code: {}, request id: {}",
                        _path.native(), outcome.GetError().GetMessage(),
                        outcome.GetError().GetExceptionName(), outcome.GetError().GetResponseCode(),
                        outcome.GetError().GetRequestId());
            }
        }
        *bytes_read = outcome.GetResult().GetContentLength();
        if (*bytes_read != bytes_req) {
            return Status::InternalError("failed to read (bytes read: {}, bytes req: {})",
                                         *bytes_read, bytes_req);
        }
        s3_bytes_read_total << bytes_req;
        s3_bytes_per_read << bytes_req;
        DorisMetrics::instance()->s3_bytes_read_total->increment(bytes_req);
        if (retry_count > 0) {
            LOG(INFO) << fmt::format("read s3 file {} succeed after {} times with {} ms sleeping",
                                     _path.native(), retry_count, total_sleep_time);
        }
        return Status::OK();
    }
    return Status::InternalError("failed to read from s3, exceeded maximum retries");
}

void S3FileReader::_collect_profile_before_close() {
    if (_profile != nullptr) {
        // nothing
    }
}

} // namespace io
} // namespace doris
