/**
 * @author Jorge Bay Gondra
 */
import { RemoteConnection } from '../driver/remote-connection.cjs';
import { Traversal } from './traversal.cjs';
export declare class TraversalStrategies {
    readonly strategies: TraversalStrategy[];
    /**
     * Creates a new instance of TraversalStrategies.
     * @param {TraversalStrategies} [parent] The parent strategies from where to clone the values from.
     * @constructor
     */
    constructor(parent?: TraversalStrategies);
    /** @param {TraversalStrategy} strategy */
    addStrategy(strategy: TraversalStrategy): void;
    /** @param {TraversalStrategy} strategy */
    removeStrategy(strategy: TraversalStrategy): TraversalStrategy | undefined;
    /**
     * @param {Traversal} traversal
     * @returns {Promise}
     */
    applyStrategies(traversal: Traversal): Promise<void>;
}
export type TraversalStrategyConfiguration = any;
export declare abstract class TraversalStrategy {
    fqcn: string;
    configuration: TraversalStrategyConfiguration;
    connection?: RemoteConnection;
    /**
     * @param {String} fqcn fully qualified class name in Java of the strategy
     * @param {TraversalStrategyConfiguration} configuration for the strategy
     */
    constructor(fqcn: string, configuration?: TraversalStrategyConfiguration);
    /**
     * @abstract
     * @param {Traversal} traversal
     * @returns {Promise}
     */
    apply(traversal: Traversal): Promise<void>;
}
export declare class ConnectiveStrategy extends TraversalStrategy {
    constructor();
}
export declare class ElementIdStrategy extends TraversalStrategy {
    constructor();
}
export declare class HaltedTraverserStrategy extends TraversalStrategy {
    /**
     * @param {String} haltedTraverserFactory full qualified class name in Java of a {@code HaltedTraverserFactory} implementation
     */
    constructor(haltedTraverserFactory: string);
}
export declare class OptionsStrategy extends TraversalStrategy {
    constructor(options: TraversalStrategyConfiguration);
}
export declare class PartitionStrategy extends TraversalStrategy {
    /**
     * @param {Object} [options]
     * @param {String} [options.partitionKey] name of the property key to partition by
     * @param {String} [options.writePartition] the value of the currently write partition
     * @param {Array<String>} [options.readPartitions] list of strings representing the partitions to include for reads
     * @param {boolean} [options.includeMetaProperties] determines if meta-properties should be included in partitioning defaulting to false
     */
    constructor(options: TraversalStrategyConfiguration);
}
export declare class SubgraphStrategy extends TraversalStrategy {
    /**
     * @param {Object} [options]
     * @param {GraphTraversal} [options.vertices] name of the property key to partition by
     * @param {GraphTraversal} [options.edges] the value of the currently write partition
     * @param {GraphTraversal} [options.vertexProperties] list of strings representing the partitions to include for reads
     * @param {boolean} [options.checkAdjacentVertices] enables the strategy to apply the {@code vertices} filter to the adjacent vertices of an edge.
     */
    constructor(options: TraversalStrategyConfiguration);
}
export declare class ProductiveByStrategy extends TraversalStrategy {
    /**
     * @param {Object} [options]
     * @param {Array<String>} [options.productiveKeys] set of keys that will always be productive
     */
    constructor(options: TraversalStrategyConfiguration);
}
export declare class VertexProgramStrategy extends TraversalStrategy {
    constructor(options: TraversalStrategyConfiguration);
}
export declare class MatchAlgorithmStrategy extends TraversalStrategy {
    /**
     * @param matchAlgorithm
     */
    constructor(matchAlgorithm: string);
}
export declare class AdjacentToIncidentStrategy extends TraversalStrategy {
    constructor();
}
export declare class FilterRankingStrategy extends TraversalStrategy {
    constructor();
}
export declare class IdentityRemovalStrategy extends TraversalStrategy {
    constructor();
}
export declare class IncidentToAdjacentStrategy extends TraversalStrategy {
    constructor();
}
export declare class InlineFilterStrategy extends TraversalStrategy {
    constructor();
}
export declare class LazyBarrierStrategy extends TraversalStrategy {
    constructor();
}
export declare class MatchPredicateStrategy extends TraversalStrategy {
    constructor();
}
export declare class OrderLimitStrategy extends TraversalStrategy {
    constructor();
}
export declare class PathProcessorStrategy extends TraversalStrategy {
    constructor();
}
export declare class PathRetractionStrategy extends TraversalStrategy {
    constructor();
}
export declare class CountStrategy extends TraversalStrategy {
    constructor();
}
export declare class RepeatUnrollStrategy extends TraversalStrategy {
    constructor();
}
export declare class GraphFilterStrategy extends TraversalStrategy {
    constructor();
}
export declare class EarlyLimitStrategy extends TraversalStrategy {
    constructor();
}
export declare class LambdaRestrictionStrategy extends TraversalStrategy {
    constructor();
}
export declare class ReadOnlyStrategy extends TraversalStrategy {
    constructor();
}
export declare class EdgeLabelVerificationStrategy extends TraversalStrategy {
    /**
     * @param {boolean} logWarnings determines if warnings should be written to the logger when verification fails
     * @param {boolean} throwException determines if exceptions should be thrown when verifications fails
     */
    constructor(logWarnings?: boolean, throwException?: boolean);
}
export declare class ReservedKeysVerificationStrategy extends TraversalStrategy {
    /**
     * @param {boolean} logWarnings determines if warnings should be written to the logger when verification fails
     * @param {boolean} throwException determines if exceptions should be thrown when verifications fails
     * @param {Array<String>} keys the list of reserved keys to verify
     */
    constructor(logWarnings?: boolean, throwException?: boolean, keys?: string[]);
}
export type SeedStrategyOptions = {
    seed: number;
};
export declare class SeedStrategy extends TraversalStrategy {
    /**
     * @param {SeedStrategyOptions} [options]
     * @param {number} [options.seed] the seed to provide to the random number generator for the traversal
     */
    constructor(options: SeedStrategyOptions);
}
