## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  eval = TRUE,
  echo = TRUE,
  comment = "#>",
  dpi = 120,
  fig.align = "center",
  out.width = "80%"
)

## ----setup--------------------------------------------------------------------
library(forcis)

## ----'download-db', eval=FALSE------------------------------------------------
# # Create a data/ folder ----
# dir.create("data")
# 
# # Download latest version of the database ----
# download_forcis_db(path = "data", version = NULL)

## ----'load-data', echo=FALSE--------------------------------------------------
file_name <- system.file(
  file.path("extdata", "FORCIS_net_sample.csv"),
  package = "forcis"
)
net_data <- read.csv(file_name)

## ----'load-data-user', eval=FALSE---------------------------------------------
# # Import net data ----
# net_data <- read_plankton_nets_data(path = "data")

## ----'select-taxo'------------------------------------------------------------
# Select taxonomy ----
net_data_vt <- net_data |>
  select_taxonomy(taxonomy = "VT")

net_data_vt

## ----'select-columns'---------------------------------------------------------
# Remove not required columns (optional) ----
net_data_vt <- net_data_vt |>
  select_forcis_columns()

net_data_vt

## ----'filter-by-month'--------------------------------------------------------
# Filter data by sampling month ----
net_data_vt_july_aug <- net_data_vt |>
  filter_by_month(months = 7:8)

# Number of original records ----
nrow(net_data_vt)

# Number of filtered records ----
nrow(net_data_vt_july_aug)

## ----'filter-by-year'---------------------------------------------------------
# Filter data by sampling year ----
net_data_vt_9020 <- net_data_vt |>
  filter_by_year(years = 1990:2020)

# Number of original records ----
nrow(net_data_vt)

# Number of filtered records ----
nrow(net_data_vt_9020)

## ----'filter-by-bbox'---------------------------------------------------------
# Filter by spatial bounding box ----
net_data_vt_bbox <- net_data_vt |>
  filter_by_bbox(bbox = c(45, -61, 82, -24))

# Number of original records ----
nrow(net_data_vt)

# Number of filtered records ----
nrow(net_data_vt_bbox)

## ----'check-bbox'-------------------------------------------------------------
# Filter by spatial bounding box ----
net_data_vt_sf <- net_data_vt |>
  data_to_sf()

net_data_vt_bbox_sf <- net_data_vt_bbox |>
  data_to_sf()

# Original spatial extent ----
sf::st_bbox(net_data_vt_sf)

# Spatial extent of filtered records ----
sf::st_bbox(net_data_vt_bbox_sf)

## ----'filter-by-ocean'--------------------------------------------------------
# Filter by ocean name ----
net_data_vt_indian <- net_data_vt |>
  filter_by_ocean(ocean = "Indian Ocean")

# Number of original records ----
nrow(net_data_vt)

# Number of filtered records ----
nrow(net_data_vt_indian)

## ----'get-ocean-names'--------------------------------------------------------
# Get ocean names ----
get_ocean_names()

## ----'filter-by-polygon'------------------------------------------------------
# Import spatial polygon ----
file_name <- system.file(
  file.path("extdata", "IHO_Indian_ocean_polygon.gpkg"),
  package = "forcis"
)

indian_ocean <- sf::st_read(file_name, quiet = TRUE)

# Filter by polygon ----
net_data_vt_poly <- net_data_vt |>
  filter_by_polygon(polygon = indian_ocean)

# Number of original records ----
nrow(net_data_vt)

# Number of filtered records ----
nrow(net_data_vt_poly)

## ----'filter-by-species'------------------------------------------------------
# Filter by species ----
net_data_vt_glutinata_nitida <- net_data_vt |>
  filter_by_species(species = c("g_glutinata_VT", "c_nitida_VT"))

# Dimensions of original data ----
dim(net_data_vt)

# Dimensions of filtered data ----
dim(net_data_vt_glutinata_nitida)

## ----'filter-counts'----------------------------------------------------------
# Keep samples with positive counts ----
net_data_vt_glutinata_nitida <- net_data_vt_glutinata_nitida |>
  dplyr::filter(g_glutinata_VT > 0 | c_nitida_VT > 0)

# Number of filtered records ----
nrow(net_data_vt_glutinata_nitida)

## ----'reshape-data'-----------------------------------------------------------
# Convert to long format ----
net_data_long <- convert_to_long_format(net_data)

# Dimensions of original data ----
dim(net_data)

# Dimensions of reshaped data ----
dim(net_data_long)

## ----'reshape-data-2'---------------------------------------------------------
# Column names ----
colnames(net_data_long)

