% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmtestspsur.R
\name{lmtestspsur}
\alias{lmtestspsur}
\title{Testing for the presence of spatial effects in Seemingly Unrelated Regressions}
\usage{
lmtestspsur(Form = NULL, data = NULL, W = NULL, X = NULL,
  Y = NULL, time = NULL, G = NULL, N = NULL, Tm = NULL,
  print_table = TRUE)
}
\arguments{
\item{Form}{An object created with the package \code{\link[Formula]{Formula}}
that describes the model to be estimated. This model may contain several
responses (explained variables) and a varying number of regressors in each equation.}

\item{data}{An object of class data.frame or a matrix.}

\item{W}{A spatial weighting matrix of order \emph{(NxN)}, assumed to be the
same for all equations and time periods.}

\item{X}{A data matrix of order \emph{(NTmGxp)} with the observations of the regressors
The number of covariates in the SUR model is p = \eqn{sum(p_{g})} where \emph{\eqn{p_{g}}}
is the number of regressors (including the intercept) in the g-th equation, \emph{g = 1,...,G}).
The specification of \emph{X} is only necessary if not available a \code{\link[Formula]{Formula}}
and a data frame. Default = \code{NULL}.}

\item{Y}{A column vector of order \emph{(NTmGx1)}, with the observations of the
explained variables. The ordering of the data must be (first) equation,
(second) time dimension and (third) Cross-sectional/spatial units.
The specification of \emph{Y} is only necessary if not available a \code{\link[Formula]{Formula}}
and a data frame. Default = \code{NULL}.}

\item{time}{Time variable.}

\item{G}{Number of equations.}

\item{N}{Number of cross-section or spatial units}

\item{Tm}{Number of time periods.}

\item{print_table}{Logical value to print the output. Default = \code{TRUE}}
}
\value{
A list including:
  \tabular{ll}{
  \code{stat_names}   \tab Name of Lagrange Multiplier. \cr
  \code{stat}   \tab Value of the corresponding Lagrange Multiplier. \cr
  \code{df} \tab Degrees of freedom for each Multiplier.\cr
  }
}
\description{
The function \code{\link{spsurml}}  reports a collection of Lagrange Multipliers
 designed to test  for the presence of different forms of spatial dependence in a
 \emph{SUR} model of the \strong{"sim"} type. That is, the approach of this function is
 from \emph{'specific to general'}. As said, the model of the null hypothesis is the \strong{"sim"}
 model whereas the model of the alternative depends on the effect whose omission we want to test.

The collection of Lagrange Multipliers obtained by \code{lmtestspsur} are standard in the
literature and take into account the multivariate nature of the \emph{SUR} model. As a limitation,
note that each Multiplier tests for the omission of the same spatial effects in all the cross-sections of
the \emph{G} equations.
}
\details{
\code{\link{lmtestspsur}} tests for the omission of spatial effects in the \strong{"sim"} version
 of the \emph{SUR} model: \cr

    \deqn{y_{tg} = X_{tg} \beta_{g} + u_{tg}}
    \deqn{E[u_{tg}u_{th}']= \sigma_{gh}I_{N}  \quad E[u_{tg}u_{sh}']= 0 \mbox{ if } t ne s}

where \eqn{y_{tg}} and \eqn{u_{tg}} are \emph{(Nx1)} vectors, corresponding to the g-th equation and time period t;
\eqn{X_{tg}} is the matrix of exogenous variables, of order\emph{\eqn{(Nxp_{g})}}. Moreover, \eqn{\beta_{g}} is an unknown
\emph{\eqn{(p_{g}x1)}} vector of coefficients and \eqn{\sigma_{gh}I_{N}} the covariance between equations \emph{g} and \emph{h},
being \eqn{\sigma_{gh}} and scalar and \eqn{I_{N}} the identity matrix of orden N.


The Lagrange Multipliers reported by this function are the followings:

  \itemize{
    \item \strong{LM-SUR-LAG}: Tests for the omission of a spatial lag of the explained variable
     in the right hand side of the \strong{"sim"} equation. The model of the alternative is:\cr

    \eqn{y_{tg} = \lambda_{g}Wy_{tg}+X_{tg} \beta_{g} + u_{tg}}

      The null and alternative hypotheses are:

         \eqn{H_{0}: \lambda_{g}=0 (forall g)} vs  \eqn{H_{A}: \lambda_{g} ne 0 (exist g)}

     \item \strong{LM-SUR-ERR}: Tests for the omission of spatial dependence in the equation of the errors
     of the \strong{"sim"} model. The model of the alternative is:

    \eqn{y_{tg} = X_{tg} \beta_{g} + u_{tg}}; \eqn{u_{tg}= \rho_{g}Wu_{tg}+\epsilon_{tg}}

      The null and alternative hypotheses are:

         \eqn{H_{0}: \rho_{g}=0 (forall g)} vs  \eqn{H_{A}: \rho_{g}  ne 0 (exist g)}

     \item \strong{LM-SUR-SARAR}: Tests for the simultaneous omission of a spatial lag of the explained
      variable in the right hand side of the \strong{"sim"} equation and spatial dependence in the
      equation of the errors. The model of the alternative is:


    \eqn{y_{tg} = \lambda_{g}Wy_{tg}+X_{tg} \beta_{g} + u_{tg}}; \eqn{u_{tg}= \rho_{g}Wu_{tg}+\epsilon_{tg}}

      The null and alternative hypotheses are:

     \eqn{H_{0}: \lambda_{g}=\rho_{g}=0 (forall g)} vs  \eqn{H_{A}: \lambda_{g} ne 0 or \rho_{g} ne 0 (exist g)}

     \item
     \strong{LM*-SUR-SLM} and \strong{LM*-SUR-SEM}: These two test are the robustifyed version of the original,
     raw Multipliers, \strong{LM-SUR-SLM} and \strong{LM-SUR-SEM}, which can be severely oversized if
     the respective alternative hypothesis is misspeficied (this would be the case if, for example, we are
     testing for omitted lags of the explained variable whereas the problem is that there is spatial dependence
     in the errors, or viceversa). The null and alternative hypotheses of both test are totally analogous to  their twin
     non robust Multipliers.
    }
}
\examples{

#################################################
######## CROSS SECTION DATA (G>1; Tm=1) # #######
#################################################

#### Example 1: Spatial Phillips-Curve. Anselin (1988, p. 203)
rm(list = ls()) # Clean memory
data("spc")
Tformula <- WAGE83 | WAGE81 ~ UN83 + NMR83 + SMSA | UN80 + NMR80 + SMSA
LMs <- lmtestspsur(Form = Tformula, data = spc, W = Wspc)

\dontrun{
#################################################
######## PANEL DATA (G>1; Tm>1)          ########
#################################################

#### Example 2: Homicides & Socio-Economics (1960-90)
# Homicides and selected socio-economic characteristics for
# continental U.S. counties.
# Data for four decennial census years: 1960, 1970, 1980 and 1990.
# https://geodacenter.github.io/data-and-lab/ncovr/

data("NCOVR")
# With different number of exogenous variables in each equation
Tformula <- HR70 | HR80  | HR90 ~ PS70 + UE70 | PS80 + UE80 +RD80 |
            PS90 + UE90 + RD90 + PO90
LMs <- lmtestspsur(Form = Tformula, data = NCOVR, W = W)
}

################################################################
######## PANEL DATA: TEMPORAL CORRELATIONS (G=1; Tm>1) ########
################################################################
\donttest{
#### Example 3: NCOVR in panel data form
## Takes less than 1 minute
data("NCOVR")
Year <- as.numeric(kronecker(c(1960,1970,1980,1990),matrix(1,nrow = dim(NCOVR)[1])))
HR <- c(NCOVR$HR60,NCOVR$HR70,NCOVR$HR80,NCOVR$HR90)
PS <- c(NCOVR$PS60,NCOVR$PS70,NCOVR$PS80,NCOVR$PS90)
UE <- c(NCOVR$UE60,NCOVR$UE70,NCOVR$UE80,NCOVR$UE90)
NCOVRpanel <- as.data.frame(cbind(Year,HR,PS,UE))
Tformula <- HR ~ PS + UE
LM_time <- lmtestspsur(Form = Tformula, data = NCOVRpanel, time = Year, W = W)
}

}
\references{
\itemize{
    \item Mur, J., López, F., and Herrera, M. (2010). Testing for spatial
      effects in seemingly unrelated regressions.
      \emph{Spatial Economic Analysis}, 5(4), 399-440.
     \item López, F.A., Mur, J., and Angulo, A. (2014). Spatial model
       selection strategies in a SUR framework. The case of regional
       productivity in EU. \emph{Annals of Regional Science}, 53(1),
       197-220.
  }
}
\seealso{
\code{\link{spsurml}}, \code{\link{lrtestspsur}}
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Jesús Mur  \tab \email{jmur@unizar.es} \cr
  }
}
