% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/concordance_check.R
\name{concordance_check}
\alias{concordance_check}
\title{Perform Concordance Check on Data Based on Defined Rules}
\usage{
concordance_check(
  S_data,
  M_data,
  Result = FALSE,
  show_column = NULL,
  date_parser_fun = smart_to_gregorian_vec,
  var_select = "all",
  verbose = FALSE
)
}
\arguments{
\item{S_data}{data.frame. The source data in which rules will be evaluated. Each column may be referenced by the rules.}

\item{M_data}{data.frame. Metadata describing variables and their concordance rules. Must include at least columns \code{VARIABLE} and \code{Concordance_Rule}. Optionally includes \code{TYPE} and \code{Concordance_Error_Type}.}

\item{Result}{logical (default: \code{FALSE}). If \code{TRUE}, returns row-by-row evaluation results for each rule. If \code{FALSE}, returns a summary table for each rule.}

\item{show_column}{character vector (default: \code{NULL}). Names of columns from \code{S_data} to include in the result when \code{Result = TRUE}. Ignored otherwise.}

\item{date_parser_fun}{function (default: \code{smart_to_gregorian_vec}). Converting Persian dates to English,Function to convert date values or date literals to \code{Date} class. Must accept character vectors and return \code{Date} objects.}

\item{var_select}{character, numeric, or \code{"all"} (default: \code{"all"}). Subset of variables (rules) to check. Can be a character vector of variable names, numeric vector of row indices in \code{M_data}, or \code{"all"} to run all rules.}

\item{verbose}{logical (default: \code{FALSE}). If \code{TRUE}, prints diagnostic messages during rule processing and evaluation.}
}
\value{
If \code{Result = FALSE}: a data.frame summary with columns:
\itemize{
  \item VARIABLE: Name of the variable/rule.
  \item Condition_Met: Number of rows where the rule is TRUE.
  \item Condition_Not_Met: Number of rows where the rule is FALSE.
  \item NA_Count: Number of rows with missing/indeterminate result.
  \item Total_Applicable: Number of non-NA rows.
  \item Total_Rows: Number of total rows.
  \item Percent_Met: Percentage of applicable rows meeting the condition.
  \item Percent_Not_Met: Percentage of applicable rows not meeting the condition.
  \item Concordance_Error_Type: Error type from metadata (if available).
}
}
\description{
This function evaluates a source dataframe (`S_data`) against a set
of rules defined in a metadata dataframe (`M_data`).
It Checks multiple concordance rules (logical and clinical conditions) on columns of a data frame, based on metadata specifications.
Supports flexible rule definition, date handling, and customizable output.
}
\details{
The metadata data.frame (\code{M_data}) must contain at least the following columns:
\itemize{
  \item \strong{VARIABLE}: The name of the variable in \code{S_data} to which the rule applies.
  \item \strong{Concordance_Rule}: The logical or clinical rule (as a string) to be evaluated for each row.
  \item \strong{TYPE}: The expected type of the variable (e.g., "numeric", "date", "character").
  \item \strong{Concordance_Error_Type}: The error type for each rule will be reported in the summary output.Based on the importance and severity of the rule, it can include two options: "Warning" or "Error".
}

For each variable described in \code{M_data}, the function:
\itemize{
  \item Replaces any instance of the string "val" in the rule with the actual column name of the variable.
  \item Parses and detects any date literals in the rule and substitutes them with placeholders; these placeholders are converted to Date class using the provided \code{date_parser_fun}.
  \item Automatically converts any referenced data columns to the appropriate type (numeric, date, or character) based on the \code{TYPE} column in the metadata.
  \item Detects which columns from \code{S_data} are referenced in each rule and ensures they are available and correctly typed before evaluation.
  \item Evaluates the rule for each row of \code{S_data}, using vectorized evaluation for performance where possible, and falling back to row-wise evaluation if necessary (e.g., for rules that are not vectorizable, such as those using \code{ifelse} with NA logic).
}

The function supports flexible rule definitions, including conditions involving multiple columns,clinical rules, date comparisons, and custom logic using R expressions.

If \code{Result = FALSE}, the function returns a summary table for each rule, including counts and percentages of rows that meet or do not meet the condition, as well as the error type from the metadata.

If \code{Result = TRUE}, the function returns a data.frame with one column per rule/variable, each containing logical values (\code{TRUE}, \code{FALSE}, or \code{NA}) for every row, plus any extra columns from \code{S_data} listed in \code{show_column}.
}
\examples{
# build the long rule in multiple short source lines to avoid >100 char Rd lines
rule_bp <- paste0(
  "(ifelse(is.na(val) | is.na(Systolic_BP2), NA, ",
  "(abs(val - Systolic_BP2) >= 15) & (val > 15 & Prescription_drug == '')))"
)

# Source data
S_data <- data.frame(
  National_code = c("123", "1456", "789","545","4454","554"),
  LastName = c("Aliyar","Johnson","Williams","Brown","Jones","Garcia"),
  VisitDate = c("2025-09-23", "2021-01-10", "2021-01-03","1404-06-28","1404-07-28",NA),
  Test_date = c("1404-07-01", "2021-01-09", "2021-01-14","1404-06-29","2025-09-19",NA),
  Certificate_validity = c("2025-09-21", "2025-01-12", "2025-02-11","1403-06-28","2025-09-19",NA),
  Systolic_BP1 = c(110, NA, 145, 125,114,NA),
  Systolic_BP2 = c(125, 150, NA, 110,100,NA),
  Prescription_drug= c("Atorvastatin", "Metformin", "Amlodipine",
    "Omeprazole", "Aspirin","Metoprolol"),
  Blood_type = c("A-","B+","AB","A+","O-","O+"),
  stringsAsFactors = FALSE
)

# META DATA (use the short-built rule)
M_data <- data.frame(
  VARIABLE = c("National_code", "Certificate_validity", "VisitDate",
               "Test_date","LastName","Systolic_BP1","Systolic_BP2",
               "Prescription_drug","Blood_type"),
  Concordance_Rule = c(
    "", "", "VisitDate<=Test_date", "Test_date-VisitDate < 7", "",
    rule_bp, "", "", ""
  ),
  TYPE=c("numeric","date","date","date","character",
         "numeric","numeric","character","character"),
  Concordance_Error_Type = c("type1",NA,"type2","type3",NA,NA,NA,NA,"type4"),
  stringsAsFactors = FALSE
)

# test call
result <- concordance_check(S_data = S_data, M_data = M_data, Result = TRUE,
show_column = c("National_code"))
print(result)
}
