#' @name FanPC.EFM
#' @title Apply the FanPC method to the Elliptical Factor Model
#' @description This function performs Factor Analysis via Principal Component (FanPC) on a given data set. It calculates the estimated factor loading matrix (AF), specific variance matrix (DF), and the mean squared errors.
#' @param data A matrix of input data.
#' @param m The number of principal components.
#' @param A The true factor loadings matrix.
#' @param D The true uniquenesses matrix.
#' @param p The number of variables.
#' @return A list containing:
#' \item{AF}{Estimated factor loadings.}
#' \item{DF}{Estimated uniquenesses.}
#' \item{MSESigmaA}{Mean squared error for factor loadings.}
#' \item{MSESigmaD}{Mean squared error for uniquenesses.}
#' \item{LSigmaA}{Loss metric for factor loadings.}
#' \item{LSigmaD}{Loss metric for uniquenesses.}
#' @examples
#' \donttest{
#' library(matrixcalc)
#' library(MASS)
#' 
#' n <- 100
#' p <- 10
#' m <- 5
#' mu <- t(matrix(rep(runif(p, 0, 1000), n), p, n))
#' mu0 <- as.matrix(runif(m, 0))
#' sigma0 <- diag(runif(m, 1))
#' F_matrix <- matrix(mvrnorm(n, mu0, sigma0), nrow = n)
#' A <- matrix(runif(p * m, -1, 1), nrow = p)
#' r <- rnorm(n * p, 0, 1)
#' epsilon <- matrix(r, nrow = n)
#' D <- diag(as.vector(apply(epsilon, 2, function(x) sum(x^2))))
#' data <- mu + F_matrix %*% t(A) + epsilon
#' results <- FanPC.EFM(data, m, A, D, p)
#' print(results)
#' }
#' @export
#' @importFrom matrixcalc frobenius.norm
#' @importFrom stats cor
FanPC.EFM <- function(data, m, A, D, p) {
  if (!is.matrix(data) && !is.data.frame(data)) {
    stop("Data must be a matrix or data frame.")
  }
  if (!is.numeric(m) || m <= 0 || m > ncol(data)) {
    stop("The number of principal components 'm' must be a positive integer not exceeding the number of columns in data.")
  }
  
  X <- scale(data)
  n <- nrow(X)
  SigmahatF <- cor(X)
  eig <- eigen(SigmahatF)
  lambdahat <- eig$values[1:m]
  ind <- order(lambdahat, decreasing = TRUE)
  lambdahat <- lambdahat[ind]
  Q <- eig$vectors
  Q <- Q[, ind]
  AF <- Q[, 1:m]
  
  hF <- diag(AF %*% t(AF))
  DF <- diag(SigmahatF - hF)
  
  MSESigmaA = frobenius.norm(AF - A)^2 / (p^2)
  MSESigmaD = frobenius.norm(DF - D)^2 / (p^2)
  LSigmaA = frobenius.norm(AF - A)^2 / frobenius.norm(A)^2
  LSigmaD = frobenius.norm(DF - D)^2 / frobenius.norm(D)^2
  
  return(list(AF = AF,
              DF = DF,
              MSESigmaA = MSESigmaA,
              MSESigmaD = MSESigmaD,
              LSigmaA = LSigmaA,
              LSigmaD = LSigmaD))
}