#' Compute the U-Matrix for a trained Gower-SOM
#'
#' Calculates the U-Matrix (unified distance matrix) for visualizing the topology
#' and cluster structure of a Self-Organizing Map trained on mixed-attribute data.
#' The U-Matrix stores, for each neuron, the average Gower distance between its
#' prototype (codebook vector) and the prototypes of its immediate neighbors.
#'
#' @details
#' The function assumes a rectangular grid topology, where each neuron has up to
#' four direct neighbors (up, down, left, right). For each neuron, the average
#' Gower distance to valid neighbors is computed and stored in the U-Matrix.
#'
#' @param codebook A data.frame or matrix containing the prototypes (weights) of
#'   the SOM after training, with one row per neuron.
#' @param n_rows Integer, number of rows in the SOM grid.
#' @param n_cols Integer, number of columns in the SOM grid.
#'
#' @return A numeric matrix of size \code{n_rows x n_cols}, where each entry
#'   contains the average distance between the corresponding neuron and its
#'   neighbors.
#'
#' @examples
#' \dontrun{
#' set.seed(1)
#' df <- data.frame(
#'   x1 = rnorm(20),
#'   x2 = rnorm(20),
#'   g  = factor(sample(letters[1:3], 20, TRUE))
#' )
#' fit <- gsom_Training(df, grid_rows = 3, grid_cols = 3,
#'                 num_iterations = 5, batch_size = 4)
#' U <- gsom_Umatrix(fit$weights, n_rows = 3, n_cols = 3)
#' image(U, main = "Gower-SOM U-Matrix")
#' }
#'
#' @seealso \code{\link[cluster]{daisy}} for Gower distance computation.
#' @export
#' @importFrom cluster daisy


gsom_Umatrix <- function(codebook, n_rows, n_cols) {

  n_nodes <- nrow(codebook)

  stopifnot(n_nodes == n_rows * n_cols)

  u_matrix <- matrix(NA, nrow = n_rows, ncol = n_cols)

  pos_to_index <- function(row, col) {
    if (row < 1 || row > n_rows || col < 1 || col > n_cols) return(NA)
    return((row - 1) * n_cols + col)
  }

  # Neighbors in rectangular topology

  neighbors_offsets <- list(c(-1, 0), c(1, 0), c(0, -1), c(0, 1))

  for (i in 1:n_rows) {
    for (j in 1:n_cols) {
      current_index <- pos_to_index(i, j)
      current_vector <- codebook[current_index, , drop = FALSE]

      neighbor_dists <- c()

      for (offset in neighbors_offsets) {
        ni <- i + offset[1]
        nj <- j + offset[2]
        neighbor_index <- pos_to_index(ni, nj)

        if (!is.na(neighbor_index)) {
          neighbor_vector <- codebook[neighbor_index, , drop = FALSE]
          combined <- rbind(current_vector, neighbor_vector)

          # Gower distance calculus

          dist_val <- as.numeric(daisy(combined, metric = "gower"))[1]
          neighbor_dists <- c(neighbor_dists, dist_val)
        }
      }

      u_matrix[i, j] <- mean(neighbor_dists)
    }
  }

  return(u_matrix)

}
