% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MNM_Hurdle_AR.R
\name{MNM_Hurdle_AR}
\alias{MNM_Hurdle_AR}
\title{Fit a multi-species N-mixture (MNM) Hurdle-AR  model using Nimble}
\usage{
MNM_Hurdle_AR(
  Y = NULL,
  iterations = 60000,
  burnin = 20000,
  thin = 10,
  Xp = NULL,
  Xn = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{Y}{Array of observed counts with dimensions \verb{(R, T, S, K)}, where:
\itemize{
\item \code{R}: Number of sites.
\item \code{T}: Number of replicates.
\item \code{S}: Number of species.
\item \code{K}: Number of time periods.
}}

\item{iterations}{Number of MCMC iterations for model fitting. Default is \verb{60,000}.}

\item{burnin}{Number of initial iterations to discard as burn-in. Default is \verb{20,000}.}

\item{thin}{Thinning interval for the MCMC chains. Default is \code{10}.}

\item{Xp}{Array of detection covariates with dimensions \verb{(R, S, P1)}, where:
\itemize{
\item \code{R}: Number of sites.
\item \code{S}: Number of species.
\item \code{P1}: Number of detection probability covariates.
}}

\item{Xn}{Array of abundance covariates with dimensions \verb{(R, S, K, P2)}, where:
\itemize{
\item \code{R}: Number of sites.
\item \code{S}: Number of species.
\item \code{K}: Number of time points.
\item \code{P2}: Number of abundance covariates.
}}

\item{verbose}{Control the level of output displayed during function execution. Default is TRUE.}

\item{...}{Additional arguments passed for prior distribution specification. Supported distributions include dnorm, dexp, dgamma, dbeta, dunif, dlnorm, dbern, dpois, dbinom, dcat, dmnorm, dwish, dchisq, dinvgamma, dt, dweib, ddirch, dmulti, dmvt. Default prior distributions are:
\itemize{
\item prior_detection_probability: prior distribution for the detection probability intercept (\code{gamma}). Default is \code{'dnorm(0, 0.001)'}.
\item prior_precision: prior distribution for the precision matrix for the species-level random effect. Default is \code{'dwish(Omega[1:S,1:S], df)'}.
\item prior_mean: prior distribution for the mean of the species-level random effect (\code{mu}). Default is \code{'dnorm(0,0.001)'}.
\item prior_hurdle: prior distribution for \code{theta}, the probability of structural zero in hurdle models. Default is \code{'dbeta(1,1)'}.
\item prior_mean_AR: prior distribution for the mean of the autoregressive random effect (\code{phi}). Default is \code{'dnorm(0,0.001)'}.
\item prior_sd_AR: prior distribution for the standard deviation of the autoregressive random effect (\code{phi}). Default is \code{'dexp(1)'}.
}
See Nimble (r-nimble.org) documentation for distribution details.}
}
\value{
An object of class \code{MNM} with the following components:
\itemize{
\item summary: Summary statistics for monitored parameters, including mean, standard deviation, standard error, quantiles, effective sample size, and Rhat values.
\item n_parameters: Number of parameters in the model (useful for calculating information criteria).
\item data: Observed abundances (\code{Y}).
\item fitted_Y: Predicted values for \code{Y}. Use these for posterior predictive checks by comparing them with observed data.
\item logLik: Log-likelihood of the observed data (\code{Y}) given the model parameters.
\item n_converged: Number of parameters with successful convergence (Rhat < 1.1).
\item plot: traceplots and density plots for all monitored variables.
}
}
\description{
Fits a multi-species N-mixture model (MNM) with an autoregressive (AR-1) component and a Hurdle (zero-altered) component using Nimble.
This model is suitable for zero-inflated data and data collected over extended time periods.
}
\details{
The MNM_Hurdle_AR model extends the standard N-mixture model by incorporating:
\itemize{
\item \strong{Hurdle (zero-altered) component:} Handles zero-inflated data by modelling excess zeros separately.
\item \strong{Autoregressive (AR-1) component:} Accounts for temporal dependencies in abundance data.
}

The model is fitted to data formatted as produced by the \code{simulateData} function. Covariates affecting detection probability and abundance may also be provided. Results include posterior summaries, model diagnostics, and predictions for posterior predictive checks.
}
\note{
Ensure that the dimensions of \code{Y}, \code{Xp}, and \code{Xn} match the requirements specified above. Mismatched dimensions will result in errors during model fitting.
}
\examples{
# Example 1: Simulate data and fit the model
# Simulating example data
set.seed(42)
R <- 5  # Number of sites
T <- 10  # Number of replicates
S <- 3  # Number of species
K <- 2  # Number of time periods
P1 <- 2  # Number of detection covariates
P2 <- 3  # Number of abundance covariates

x<-simulateData(model="HurdleAR", R=R, T=T, ,S=S, K=K)
Xp <- array(runif(R * S * K * P1), dim = c(R, S, K, P1))
Xn <- array(runif(R * S * K * P2), dim = c(R, S, K, P2))
# Fit the MNM_Hurdle_AR model
\donttest{result <- MNM_Hurdle_AR(Y = x[["Y"]], Xp = Xp, Xn = Xn)}
# nimble creates auxiliary functions that may be removed after model run
# is complete using rm(list = ls(pattern = "^str"))
# Access results
\donttest{print(result@summary)}

data(birds)

# Example 2: North American Breeding Bird Data
# Data must first be reformatted to an array of dimension (R,T,S,K)
R <- 15
T <- 10
S <- 10
K <- 4
# Ensure data is ordered consistently
birds <- birds[order(birds$Route, birds$Year, birds$English_Common_Name), ]

# Create a 4D array with proper dimension
Y <- array(NA, dim = c(R, T, S, K))

# Map route, species, and year to indices
route_idx <- as.numeric(factor(birds$Route))
species_idx <- as.numeric(factor(birds$English_Common_Name))
year_idx <- as.numeric(factor(birds$Year))

# Populate the array
stop_data <- as.matrix(birds[, grep("^Stop", colnames(birds))])

for (i in seq_len(nrow(birds))) {
  Y[route_idx[i], , species_idx[i], year_idx[i]] <- stop_data[i, ]
  }

  # Assign dimnames
  dimnames(Y) <- list(
    Route = sort(unique(birds$Route)),
      Stop = paste0("Stop", 1:T),
        Species = sort(unique(birds$English_Common_Name)),
          Year = sort(unique(birds$Year))
          )

# Selecting only 5 bird species for analysis:
Y<-Y[,,1:5,]

\donttest{model<-MNM_fit(Y=Y, AR=TRUE, Hurdle=TRUE, iterations=5000, burnin=1000)}

}
\references{
\itemize{
\item Royle, J. A. (2004). N-mixture models for estimating population size from spatially replicated counts. Biometrics, 60(1), 108-115.
\item Mimnagh, N., Parnell, A., Prado, E., & Moral, R. D. A. (2022). Bayesian multi-species N-mixture models for unmarked animal communities. Environmental and Ecological Statistics, 29(4), 755-778.
}
}
\seealso{
\itemize{
\item \code{simulateData}: For generating example datasets compatible with this function.
\item \code{MNM}: For details on creating covariate arrays Xp and Xn.
}
}
