% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NADA_ken.R
\name{cenken}
\alias{cenken}
\title{Compute Kendall's Tau Correlation and ATS Line for Censored Data}
\usage{
cenken(y, ycen = NULL, x = NULL, xcen = NULL)
}
\arguments{
\item{y}{A numeric vector of observations or a formula.}

\item{ycen}{A logical vector indicating \code{TRUE} where an observation in \code{y} is
censored (a less-than value) and \code{FALSE} otherwise. Can be omitted for the case
where \code{y} is not censored.}

\item{x}{A numeric vector of observations.}

\item{xcen}{default is NULL for trend analysis purposes. If included, a
logical vector indicating \code{TRUE} where an observation in \code{x} is
censored (a less-than value) and \code{FALSE} otherwise.}
}
\value{
A list containing:
\item{tau}{Kendall's tau correlation coefficient.}
\item{slope}{The estimated slope from the ATS line.}
\item{p.value}{P-value for testing the null hypothesis of no association.}
}
\description{
Computes Kendall's tau for singly (y only) or doubly (x and y) censored data.
Also computes the Akritas-Theil-Sen (ATS) nonparametric regression line,
with the Turnbull estimate of the intercept.
}
\details{
If using the formula interface, the \code{ycen}, \code{x}, and \code{xcen}
arguments are not specified. All information is instead provided through a
formula via the \code{y} argument. The formula must use a \code{Cen} object
as the response (on the left-hand side of \code{~}), and predictors (optional)
on the right-hand side separated by \code{+}. See examples below.

Kendall's tau is a nonparametric correlation coefficient that measures
monotonic association between \code{y} and \code{x}. For left-censored data,
concordant and discordant pairs are evaluated wherever possible. For example,
with increasing \code{x} values, a change in \code{y} from <1 to 10 is considered
an increase (concordant), while a change from <1 to 0.5 or <5 is considered a tie.

The ATS line is defined as the slope resulting in Kendall's tau of 0
between residuals \code{(y - slope * x)} and \code{x}. The routine uses
an iterative bisection search to find this slope. The intercept is the
median residual, computed using the Turnbull estimate for interval-censored
data as implemented in the \pkg{Icens} package.
}
\examples{
# Both y and x are censored
data(PbHeron)
with(PbHeron,cenken(Blood,BloodCen,Kidney,KidneyCen))

# x is not censored
data(TCEReg)
with(TCEReg, cenken(log(TCEConc), TCECen, PopDensity))

# Synthetic time-series with trend analysis
set.seed(123)

## Parameters
n <- 15  # 15 years of data
time <- 1:n

## Components
trend <- 0.235 * time
noise <- rnorm(n, mean = 5, sd = 1.5)

syn_dat <- data.frame(Yr = 1989 + time, value = trend + noise)
syn_dat$censored <- syn_dat$value < quantile(syn_dat$value, 0.2)

with(syn_dat,cenken(value,censored,Yr))
\dontrun{
plot(value~Yr,syn_dat,pch=21,bg=ifelse(syn_dat$censored==TRUE,"red","blue",cex=1.5))
abline(h=quantile(syn_dat$value, 0.2),lty=2,col="red")
}

}
\references{
Helsel, D. R. (2005). \emph{Nondetects and Data Analysis: Statistics for Censored Environmental Data}.
John Wiley & Sons.

Akritas, M. G., Murphy, S. A., & LaValley, M. P. (1995).
The Theil-Sen Estimator with Doubly Censored Data and Applications to Astronomy.
\emph{Journal of the American Statistical Association}, 90, 170–177.
}
\keyword{regression}
