% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obn_udt.R
\name{ob_numerical_udt}
\alias{ob_numerical_udt}
\title{Optimal Binning for Numerical Variables using Entropy-Based Partitioning}
\usage{
ob_numerical_udt(
  feature,
  target,
  min_bins = 3,
  max_bins = 5,
  bin_cutoff = 0.05,
  max_n_prebins = 20,
  laplace_smoothing = 0.5,
  monotonicity_direction = "none",
  convergence_threshold = 1e-06,
  max_iterations = 1000
)
}
\arguments{
\item{feature}{Numeric vector of feature values. Missing values (NA) are handled
by placing them in a separate bin. Infinite values are treated as valid numeric
extremes or placed in the missing bin if they represent errors.}

\item{target}{Integer vector of binary target values (must contain only 0 and 1).
Must have the same length as \code{feature}.}

\item{min_bins}{Minimum number of bins (default: 3). Must be at least 2.}

\item{max_bins}{Maximum number of bins (default: 5). Must be greater than or
equal to \code{min_bins}.}

\item{bin_cutoff}{Minimum fraction of total observations per bin (default: 0.05).
Bins below this threshold are merged based on Event Rate similarity.}

\item{max_n_prebins}{Maximum number of pre-bins (default: 20). The algorithm
will select the top \code{max_n_prebins} cutpoints with highest Information Gain.
\strong{Performance Note}: High values (>50) may significantly slow down
processing for large datasets due to the O(N^2) nature of cutpoint selection.}

\item{laplace_smoothing}{Laplace smoothing parameter (default: 0.5) for WoE calculation.}

\item{monotonicity_direction}{String specifying monotonicity constraint:
\itemize{
  \item \code{"none"} (default): No monotonicity enforcement.
  \item \code{"increasing"}: WoE must be non-decreasing.
  \item \code{"decreasing"}: WoE must be non-increasing.
  \item \code{"auto"}: Automatically determined by Pearson correlation.
}}

\item{convergence_threshold}{Convergence threshold for IV optimization (default: 1e-6).}

\item{max_iterations}{Maximum iterations for optimization loop (default: 1000).}
}
\value{
A list containing:
\describe{
  \item{id}{Integer bin identifiers (1-based).}
  \item{bin}{Character bin intervals \code{"(lower;upper]"}.}
  \item{woe}{Numeric WoE values.}
  \item{iv}{Numeric IV contributions.}
  \item{event_rate}{Numeric event rates.}
  \item{count}{Integer total observations.}
  \item{count_pos}{Integer positive class counts.}
  \item{count_neg}{Integer negative class counts.}
  \item{cutpoints}{Numeric bin boundaries.}
  \item{total_iv}{Total Information Value.}
  \item{gini}{Gini index (2*AUC - 1) calculated on the binned data.}
  \item{ks}{Kolmogorov-Smirnov statistic calculated on the binned data.}
  \item{converged}{Logical convergence flag.}
  \item{iterations}{Integer iteration count.}
}
}
\description{
Implements a supervised binning algorithm that uses Information Gain (Entropy)
to identify the most informative initial split points, followed by a bottom-up
merging process to satisfy constraints (minimum frequency, monotonicity, max bins).

Although historically referred to as "Unsupervised Decision Trees" in some contexts,
this method is strictly **supervised** (uses target variable) and operates
**bottom-up** after an initial entropy-based selection of cutpoints. It is
particularly effective when the relationship between feature and target is
non-linear but highly informative in specific regions.
}
\details{
\strong{Algorithm Overview}

The UDT algorithm executes in four phases:

\strong{Phase 1: Entropy-Based Pre-binning}

The algorithm evaluates every possible cutpoint \eqn{c} (midpoints between sorted
unique values) using Information Gain (IG):
\deqn{IG(S, c) = H(S) - \left( \frac{|S_L|}{|S|} H(S_L) + \frac{|S_R|}{|S|} H(S_R) \right)}

The top \code{max_n_prebins} cutpoints with the highest IG are selected to form
the initial bins. This ensures that the starting bins capture the most discriminative
regions of the feature space.

\strong{Phase 2: Rare Bin Merging}

Bins with frequency \eqn{< \text{bin\_cutoff}} are merged. The merge partner is
chosen to minimize the difference in Event Rates:
\deqn{\text{merge\_idx} = \arg\min_{j \in \{i-1, i+1\}} |ER_i - ER_j|}
This differs from IV-based methods and aims to preserve local risk probability
smoothness.

\strong{Phase 3: Monotonicity Enforcement}

If requested, monotonicity is enforced by iteratively merging bins that violate
the specified direction (\code{"increasing"}, \code{"decreasing"}, or \code{"auto"}).
Auto-direction is determined by the sign of the Pearson correlation between
feature and target.

\strong{Phase 4: Constraint Satisfaction}

If \eqn{k > \text{max\_bins}}, bins are merged minimizing IV loss until the
constraint is met.

\strong{Warning on Complexity}

The pre-binning phase evaluates Information Gain for \emph{all} unique values.
For continuous features with many unique values (e.g., \eqn{N > 10,000}), this
step can be computationally intensive (\eqn{O(N^2)}). Consider rounding or using
\code{\link{ob_numerical_sketch}} for very large datasets.
}
\references{
\itemize{
  \item Quinlan, J. R. (1986). "Induction of Decision Trees". \emph{Machine Learning}, 1(1), 81-106.
  \item Fayyad, U. M., & Irani, K. B. (1992). "On the Handling of Continuous-Valued Attributes in Decision Tree Generation". \emph{Machine Learning}, 8, 87-102.
  \item Liu, H., et al. (2002). "Discretization: An Enabling Technique". \emph{Data Mining and Knowledge Discovery}, 6(4), 393-423.
}
}
\seealso{
\code{\link{ob_numerical_mdlp}} for a pure MDL-based approach,
\code{\link{ob_numerical_sketch}} for fast approximation on large data.
}
\author{
Lopes, J. E.
}
