% Generated by roxygen2: do not edit by hand
% Please edit documentation in the corresponding C++ file
\name{computeComplexPolynomial}
\alias{computeComplexPolynomial}
\title{Compute Complex Polynomial Coefficients from a Persistence Diagram}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeComplexPolynomial()}
computes the coefficients of a complex polynomial
\deqn{C_X(z) = \prod_{i=1}^N [z - X(b_i,d_i)],}
where \eqn{X:\mathbb{R}^2\rightarrow\mathbb{C}} is any one of the following three functions:

\itemize{
  \item \eqn{R(x,y)=x+iy.}
  \item  \eqn{S(x, y) =
\begin{cases}
\frac{y - x}{\alpha \sqrt{2}}(x+iy) & \text{if } (x, y) \neq (0, 0) \\
0 & \text{otherwise.}
\end{cases}}
  \item \eqn{T(x, y) = \frac{y - x}{2} [(\cos\alpha - \sin\alpha)+i(\cos\alpha + \sin\alpha)]},
}
where \eqn{\alpha=\sqrt{x^2+y^2}}. Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computeComplexPolynomial(D, homDim, m = 1, polyType = "R")
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}

\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}

\item{m}{the number of coefficients to return (default is 1). Must be less than or equal to the number of points in the diagram.}

\item{polyType}{a string specifying the polynomial type to use. Options are "R" (default), "S", or "T".}
}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes values based on the filtered data and \code{polyType}. 
If \code{D} does not contain any points corresponding to \code{homDim}, a matrix of zeros is returned.
}

\value{
A numeric matrix of dimension \eqn{m \times 2}. Each row corresponds to a coefficient of the polynomial \eqn{C_X(z)}:
\itemize{
  \item Column 1: Real part of the coefficient.
  \item Column 2: Imaginary part of the coefficient.
}
}

\author{
Umar Islambekov
}

\references{
1. Ferri, M. and Landi, C., (1999). Representing size functions by complex polynomials. Proc. Math. Met. in Pattern Recognition, 9, pp.16-19.

2. Di Fabio, B. and Ferri, M., (2015). Comparing persistence diagrams through complex vectors. In Image Analysis and Processing—ICIAP 2015:
18th International Conference, Genoa, Italy, September 7-11, 2015, Proceedings, Part I 18 (pp. 294-305). Springer International Publishing.

3. Ali, D., Asaad, A., Jimenez, M.J., Nanda, V., Paluzo-Hidalgo, E. and Soriano-Trigueros, M., (2023). A survey of vectorization methods
in topological data analysis. IEEE Transactions on Pattern Analysis and Machine Intelligence.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

# Compute first 5 coefficients of the complex polynomial of type "R" for dimension H_0
computeComplexPolynomial(D, homDim = 0, m = 5, polyType = "R")

# Compute the first 5 coefficients of the complex polynomial of type "S"
#  for homological dimension H_0
computeComplexPolynomial(D, homDim = 0, m = 5, polyType = "S")
}
