% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/export.R
\name{export_abnFit}
\alias{export_abnFit}
\title{Export abnFit object to structured JSON format}
\usage{
export_abnFit(
  object,
  format = "json",
  include_network = TRUE,
  file = NULL,
  pretty = TRUE,
  scenario_id = NULL,
  label = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{abnFit}, typically created by \code{\link{fitAbn}}.}

\item{format}{Character string specifying the export format. Currently, only
\code{"json"} is supported.}

\item{include_network}{Logical, whether to include network structure (variables
and arcs). Default is \code{TRUE}.}

\item{file}{Optional character string specifying a file path to save the JSON
output. If \code{NULL} (default), the JSON string is returned.}

\item{pretty}{Logical, whether to format the JSON output with indentation for
readability. Default is \code{TRUE}. Set to \code{FALSE} for more compact output.}

\item{scenario_id}{Optional character string or numeric identifier for the model
run or scenario. Useful for tracking multiple model versions or experiments.
Default is \code{NULL}.}

\item{label}{Optional character string providing a descriptive name or label
for the scenario. Default is \code{NULL}.}

\item{...}{Additional export options (currently unused, reserved for future extensions).}
}
\value{
If \code{file} is \code{NULL}, returns a character string containing the JSON
representation of the model. If \code{file} is provided, writes the JSON to
the specified file and invisibly returns the file path.
}
\description{
Exports a fitted Additive Bayesian Network (ABN) model to a structured JSON
format suitable for storage, sharing, and interoperability with other analysis
tools. The export includes network structure (variables and arcs) and model
parameters (coefficients, variances, and their associated metadata).
}
\details{
This function provides a standardized way to export fitted ABN models to JSON,
facilitating model sharing, archiving, and integration with external tools or
databases. The JSON structure is designed to be both human-readable and
machine-parseable, following a flat architecture to avoid deep nesting.
\subsection{Supported Model Types}{

The function handles different model fitting methods:
\itemize{
\item \strong{MLE without grouping}: Standard maximum likelihood estimation
for all supported distributions (Gaussian, Binomial, Poisson, Multinomial).
Exports fixed-effect parameters with standard errors.
\item \strong{MLE with grouping}: Generalized Linear Mixed Models (GLMM)
with group-level random effects. Exports both fixed effects (mu, betas)
and random effects (sigma, sigma_alpha).
\item \strong{Bayesian}: Placeholder for future implementation of Bayesian
model exports including posterior distributions.
}
}

\subsection{JSON Structure Overview}{

The exported JSON follows a three-component structure:
\itemize{
\item \strong{variables}: An array where each element represents a node/variable
in the network with metadata including identifier, attribute name, distribution
type, and states (for categorical variables).
\item \strong{parameters}: An array where each element represents a model
parameter (intercepts, coefficients, variances) with associated values,
standard errors, link functions, and parent variable conditions.
\item \strong{arcs}: An array where each element represents a directed edge
in the network, specifying source and target variable identifiers.
}

Additionally, optional top-level fields \code{scenario_id} and \code{label}
can be used to identify and describe the model.
}
}
\section{JSON Schema}{

\subsection{Top-Level Fields}{

\describe{
\item{\code{scenario_id}}{Optional string or numeric identifier for the model
run. Can be \code{null}.}
\item{\code{label}}{Optional descriptive name for the model. Can be \code{null}.}
\item{\code{variables}}{Array of variable objects (see Variables section).}
\item{\code{parameters}}{Array of parameter objects (see Parameters section).}
\item{\code{arcs}}{Array of arc objects (see Arcs section).}
}
}

\subsection{Variables Array}{

Each variable object contains:
\describe{
\item{\code{variable_id}}{Unique identifier for the variable (string). This ID
is used throughout the JSON to reference this variable in parameters' \code{source}
fields and in arcs' \code{source_variable_id}/\code{target_variable_id} fields.}
\item{\code{attribute_name}}{Original attribute name from the data (string).}
\item{\code{model_type}}{Distribution type: \code{"gaussian"}, \code{"binomial"},
\code{"poisson"}, or \code{"multinomial"}.}
\item{\code{states}}{Array of state objects for multinomial variables only.
Each state has \code{state_id} (used to reference specific categories in
parameters), \code{value_name} (the category label), and \code{is_baseline}
(whether this is the reference category). \code{NULL} for continuous variables.}
}
}

\subsection{Parameters Array}{

Each parameter object contains:
\describe{
\item{\code{parameter_id}}{Unique identifier for the parameter (string).}
\item{\code{name}}{Parameter name (e.g., \code{"intercept"}, \code{"prob_2"},
coefficient name, \code{"sigma"}, \code{"sigma_alpha"}).}
\item{\code{link_function_name}}{Link function: \code{"identity"} (Gaussian),
\code{"logit"} (Binomial, Multinomial), or \code{"log"} (Poisson).}
\item{\code{source}}{Object identifying which variable and state this parameter
belongs to. Contains \code{variable_id} (required, references a variable from
the variables array) and optional \code{state_id} (references a specific state
for category-specific parameters in multinomial models).}
\item{\code{coefficients}}{Array of coefficient objects (typically length 1),
each with \code{value}, \code{stderr} (or \code{NULL} for mixed models),
\code{condition_type}, and \code{conditions} array.}
}
\subsection{Coefficient Condition Types}{

\itemize{
\item \code{"intercept"}: Baseline parameter with no parent dependencies
\item \code{"linear_term"}: Effect of a parent variable
\item \code{"CPT_combination"}: Conditional probability table entry (future use)
\item \code{"variance"}: Residual variance (Gaussian/Poisson only)
\item \code{"random_variance"}: Random effect variance (mixed models)
\item \code{"random_covariance"}: Random effect covariance (multinomial mixed models)
}
}

}

\subsection{Arcs Array}{

Each arc object contains:
\describe{
\item{\code{source_variable_id}}{Identifier of the parent/source node.}
\item{\code{target_variable_id}}{Identifier of the child/target node.}
}
}
}

\section{Design Rationale}{


The JSON structure uses a flat architecture with three parallel arrays rather
than deeply nested objects. This design offers several advantages:
\itemize{
\item \strong{Database compatibility}: Easy to store in relational or document
databases with minimal transformation.
\item \strong{Extensibility}: New parameter types or metadata can be added
without restructuring existing fields.
\item \strong{Parsability}: Simpler to query and transform programmatically.
\item \strong{Flexibility}: Supports both CPT-style and GLM(M)-style models
through the polymorphic \code{source} and \code{conditions} structure.
}

Parameters are linked to variables through the \code{source.variable_id} field,
with optional \code{source.state_id} for category-specific parameters in
multinomial models. Parent dependencies are encoded in the \code{conditions}
array within each coefficient.
}

\examples{
\dontrun{
# Load example data and fit a model
library(abn)
data(ex1.dag.data)

# Define distributions
mydists <- list(b1 = "binomial", p1 = "poisson", g1 = "gaussian",
                b2 = "binomial", p2 = "poisson", g2 = "gaussian",
                b3 = "binomial", g3 = "gaussian")

# Build score cache
mycache <- buildScoreCache(data.df = ex1.dag.data,
                            data.dists = mydists,
                            method = "mle",
                            max.parents = 2)

# Find most probable DAG
mp_dag <- mostProbable(score.cache = mycache)

# Fit the model
myfit <- fitAbn(object = mp_dag, method = "mle")

# Export to JSON string with metadata
json_export <- export_abnFit(myfit,
                             scenario_id = "example_model_v1",
                             label = "Example ABN Model")

# View the structure
library(jsonlite)
parsed <- fromJSON(json_export)
str(parsed, max.level = 2)

# Export to file
export_abnFit(myfit,
              file = "my_abn_model.json",
              scenario_id = "example_model_v1",
              label = "Example ABN Model",
              pretty = TRUE)

# Export with compact formatting
compact_json <- export_abnFit(myfit, pretty = FALSE)

# ---
# Mixed-effects model example
# (Requires data with grouping structure)

# Add grouping variable
ex1.dag.data$group <- rep(1:5, length.out = nrow(ex1.dag.data))

# Build cache with grouping
mycache_grouped <- buildScoreCache(data.df = ex1.dag.data,
                                    data.dists = mydists,
                                    method = "mle",
                                    group.var = "group",
                                    max.parents = 2)

# Fit grouped model
myfit_grouped <- fitAbn(object = mp_dag,
                        method = "mle",
                        group.var = "group")

# Export grouped model (includes random effects)
json_grouped <- export_abnFit(myfit_grouped,
                              scenario_id = "grouped_model_v1",
                              label = "Mixed Effects ABN")
}

}
\seealso{
\itemize{
\item \code{\link{fitAbn}} for fitting ABN models
\item \code{\link{buildScoreCache}} for structure learning
\item \code{\link{mostProbable}} for finding the most probable network structure
}
}
