# validation.R

#' Validate bias selections
#'
#' Ensures that the number of selected biases matches the declared total
#' and that no bias is assigned more than one distribution.
#'
#' @param b_types,s_types,d_types,e_types,en_types Character vectors of bias names
#' @param num_biases Integer. Total number of biases declared.
#'
#' @return Invisibly TRUE; otherwise throws an error
#' @export
validate_bias_selection <- function(
    b_types,
    s_types,
    d_types,
    e_types,
    en_types,
    num_biases
) {

  all_biases <- c(b_types, s_types, d_types, e_types, en_types)
  all_biases <- all_biases[all_biases != ""]

  if (length(all_biases) != num_biases) {
    stop(
      "Number of selected biases does not match 'num_biases'.",
      call. = FALSE
    )
  }

  if (any(duplicated(all_biases))) {
    stop(
      "A bias cannot be assigned to more than one distribution.",
      call. = FALSE
    )
  }

  invisible(TRUE)
}


#' Validate positivity constraints
#'
#' Checks that numeric inputs are strictly positive, allowing
#' for specified index exceptions.
#'
#' @param values Numeric vector
#' @param exceptions Optional integer vector of indices allowed to be non-positive
#'
#' @return Invisibly TRUE; otherwise throws an error
#' @export
validate_positive <- function(values, exceptions = NULL) {

  values <- ifelse(is.na(values), 999, values)

  if (!is.null(exceptions)) {
    values <- values[-exceptions]
  }

  if (!all(values > 0)) {
    stop(
      "Non-positive values found where positive values are required.",
      call. = FALSE
    )
  }

  invisible(TRUE)
}
