% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quick_cor.R
\name{quick_cor}
\alias{quick_cor}
\title{Quick Correlation Analysis with Heatmap Visualization}
\usage{
quick_cor(
  data,
  vars = NULL,
  method = c("pearson", "spearman", "kendall"),
  use = "pairwise.complete.obs",
  p_adjust_method = c("none", "holm", "hochberg", "hommel", "bonferroni", "BH", "BY",
    "fdr"),
  sig_level = c(0.001, 0.01, 0.05),
  type = c("full", "upper", "lower"),
  show_coef = FALSE,
  show_sig = TRUE,
  hc_order = TRUE,
  hc_method = "complete",
  palette = "gradient_rd_bu",
  lab_size = 3,
  title = NULL,
  show_axis_x = TRUE,
  show_axis_y = TRUE,
  axis_x_angle = 45,
  axis_y_angle = 0,
  axis_text_size = 10,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{A data frame containing numeric variables.}

\item{vars}{Optional character vector specifying which variables to include.
If \code{NULL} (default), all numeric columns will be used.}

\item{method}{Character. Correlation method: "pearson" (default), "spearman",
or "kendall".}

\item{use}{Character. Method for handling missing values, passed to \code{cor()}.
Default is "pairwise.complete.obs". Other options: "everything",
"all.obs", "complete.obs", "na.or.complete".}

\item{p_adjust_method}{Character. Method for p-value adjustment for multiple
testing. Default is "none". Options: "holm", "hochberg", "hommel",
"bonferroni", "BH", "BY", "fdr", "none". See \code{\link[stats]{p.adjust}}.}

\item{sig_level}{Numeric vector. Significance levels for star annotations.
Default is \code{c(0.001, 0.01, 0.05)} corresponding to ***, **, *.}

\item{type}{Character. Type of heatmap: "full" (default), "upper", or "lower".}

\item{show_coef}{Logical. Display correlation coefficients on the heatmap?
Default is \code{FALSE}.}

\item{show_sig}{Logical. Display significance stars on the heatmap?
Default is \code{TRUE}.}

\item{hc_order}{Logical. Reorder variables using hierarchical clustering?
Default is \code{TRUE}.}

\item{hc_method}{Character. Hierarchical clustering method if \code{hc_order = TRUE}.
Default is "complete". See \code{\link[stats]{hclust}}.}

\item{palette}{Character. Color palette name from evanverse palettes.
Default is "gradient_rd_bu" (diverging Red-Blue palette, recommended for
correlation matrices). Set to \code{NULL} to use ggplot2 defaults.
Other diverging options: "piyg", "earthy_diverge", "fire_ice_duo".}

\item{lab_size}{Numeric. Size of coefficient labels if \code{show_coef = TRUE}.
Default is 3.}

\item{title}{Character. Plot title. Default is \code{NULL} (no title).}

\item{show_axis_x}{Logical. Display x-axis labels? Default is \code{TRUE}.}

\item{show_axis_y}{Logical. Display y-axis labels? Default is \code{TRUE}.}

\item{axis_x_angle}{Numeric. Rotation angle for x-axis labels in degrees.
Default is 45. Common values: 0 (horizontal), 45 (diagonal), 90 (vertical).}

\item{axis_y_angle}{Numeric. Rotation angle for y-axis labels in degrees.
Default is 0 (horizontal).}

\item{axis_text_size}{Numeric. Font size for axis labels. Default is 10.}

\item{verbose}{Logical. Print diagnostic messages? Default is \code{TRUE}.}

\item{...}{Additional arguments (currently unused, reserved for future extensions).}
}
\value{
An object of class \code{quick_cor_result} containing:
\describe{
\item{plot}{A ggplot object with the correlation heatmap}
\item{cor_matrix}{Correlation coefficient matrix}
\item{p_matrix}{P-value matrix (unadjusted)}
\item{p_adjusted}{Adjusted p-value matrix (if p_adjust_method != "none")}
\item{method_used}{Correlation method used}
\item{significant_pairs}{Data frame of significant correlation pairs}
\item{descriptive_stats}{Descriptive statistics for each variable}
\item{parameters}{List of analysis parameters}
\item{timestamp}{POSIXct timestamp of analysis}
}
}
\description{
Perform correlation analysis with automatic p-value calculation and
publication-ready heatmap visualization. Supports multiple correlation
methods and significance testing with optional multiple testing correction.
}
\details{
\strong{"Quick" means easy to use, not simplified or inaccurate.}

This function performs complete correlation analysis with proper statistical testing:

\subsection{Correlation Methods}{
\itemize{
\item \strong{Pearson}: Measures linear relationships, assumes normality
\item \strong{Spearman}: Rank-based, robust to outliers and non-normality
\item \strong{Kendall}: Rank-based, better for small samples or many ties
}
}

\subsection{P-value Calculation}{
P-values are calculated for each pairwise correlation. The function
automatically uses \code{psych::corr.test()} if the \code{psych} package
is installed, which provides significantly faster computation (10-100x speedup
for large matrices) compared to the base R \code{stats::cor.test()} loop.
If \code{psych} is not available, the function gracefully falls back to the
base R implementation.

For large correlation matrices with many tests, consider using
\code{p_adjust_method} to control for multiple testing (e.g., "bonferroni"
or "fdr").

\strong{Performance tip}: Install the \code{psych} package for faster
p-value computation:
\code{install.packages("psych")}
}

\subsection{Visualization}{
The heatmap includes:
\itemize{
\item Color-coded correlation coefficients (red = positive, blue = negative)
\item Optional significance stars (***, **, *)
\item Optional coefficient values
\item Hierarchical clustering to group similar variables
\item Publication-ready styling
}
}
}
\section{Important Notes}{


\itemize{
\item \strong{Numeric variables only}: The function automatically selects
numeric columns or uses the variables specified in \code{vars}.
\item \strong{Constant variables}: Variables with zero variance are
automatically removed with a warning.
\item \strong{Sample size}: The function will warn if sample sizes are
very small (n < 5) after removing missing values.
\item \strong{Missing values}: Handled according to the \code{use} parameter.
"pairwise.complete.obs" is recommended for optimal sample size usage.
\item \strong{Optional dependencies}: For optimal performance, install
\code{psych} (fast p-value computation) and \code{ggcorrplot} (heatmap
visualization). The function will work without them but may be slower
or use fallback plotting.
}
}

\examples{
# Example 1: Basic correlation analysis
result <- quick_cor(mtcars)
print(result)

# Example 2: Spearman correlation with specific variables
result <- quick_cor(
  mtcars,
  vars = c("mpg", "hp", "wt", "qsec"),
  method = "spearman"
)

# Example 3: Upper triangular with Bonferroni correction
result <- quick_cor(
  iris,
  type = "upper",
  p_adjust_method = "bonferroni",
  show_coef = TRUE
)

# Example 4: Custom palette and title
result <- quick_cor(
  mtcars,
  palette = "gradient_rd_bu",
  title = "Correlation Matrix of mtcars Dataset",
  hc_order = TRUE
)

# Example 5: Customize axis labels
result <- quick_cor(
  mtcars,
  axis_x_angle = 90,      # Vertical x-axis labels
  axis_text_size = 12,    # Larger text
  show_axis_y = FALSE     # Hide y-axis labels
)

# Access components
result$plot                 # ggplot object
result$cor_matrix           # Correlation matrix
result$significant_pairs    # Significant pairs
summary(result)             # Detailed summary

}
\seealso{
\code{\link[stats]{cor}}, \code{\link[stats]{cor.test}}
}
