/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

#ifndef NDEBUG

#define H5EA_MAX_NELMTS_IDX_MAX 64
#endif 

#define H5EA_SBLK_DBLK_NELMTS(s, m) (size_t) H5_EXP2(((s) + 1) / 2) * (m)

typedef H5FL_fac_head_t *H5FL_fac_head_ptr_t;

H5FL_DEFINE_STATIC(H5EA_hdr_t);

H5FL_SEQ_DEFINE_STATIC(H5FL_fac_head_ptr_t);

H5FL_SEQ_DEFINE_STATIC(H5EA_sblk_info_t);

H5EA_hdr_t *
H5EA__hdr_alloc(H5F_t *f)
{
    H5EA_hdr_t *hdr       = NULL; 
    H5EA_hdr_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (hdr = H5FL_CALLOC(H5EA_hdr_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array shared header");

    
    hdr->addr = HADDR_UNDEF;

    
    hdr->f           = f;
    hdr->swmr_write  = (H5F_INTENT(f) & H5F_ACC_SWMR_WRITE) > 0;
    hdr->sizeof_addr = H5F_SIZEOF_ADDR(f);
    hdr->sizeof_size = H5F_SIZEOF_SIZE(f);

    
    ret_value = hdr;

done:
    if (!ret_value)
        if (hdr && H5EA__hdr_dest(hdr) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_init(H5EA_hdr_t *hdr, void *ctx_udata)
{
    hsize_t start_idx;  
    hsize_t start_dblk; 
    size_t  u;          
    herr_t  ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->cparam.max_nelmts_bits);
    assert(hdr->cparam.data_blk_min_elmts);
    assert(hdr->cparam.sup_blk_min_data_ptrs);

    
    hdr->nsblks           = 1 + (hdr->cparam.max_nelmts_bits - H5VM_log2_of2(hdr->cparam.data_blk_min_elmts));
    hdr->dblk_page_nelmts = (size_t)1 << hdr->cparam.max_dblk_page_nelmts_bits;
    hdr->arr_off_size     = (unsigned char)H5EA_SIZEOF_OFFSET_BITS(hdr->cparam.max_nelmts_bits);

    
    if (NULL == (hdr->sblk_info = H5FL_SEQ_MALLOC(H5EA_sblk_info_t, hdr->nsblks)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, FAIL, "memory allocation failed for super block info array");

    
    start_idx  = 0;
    start_dblk = 0;
    for (u = 0; u < hdr->nsblks; u++) {
        hdr->sblk_info[u].ndblks      = (size_t)H5_EXP2(u / 2);
        hdr->sblk_info[u].dblk_nelmts = H5EA_SBLK_DBLK_NELMTS(u, hdr->cparam.data_blk_min_elmts);
        hdr->sblk_info[u].start_idx   = start_idx;
        hdr->sblk_info[u].start_dblk  = start_dblk;

        
        start_idx += (hsize_t)hdr->sblk_info[u].ndblks * (hsize_t)hdr->sblk_info[u].dblk_nelmts;
        start_dblk += (hsize_t)hdr->sblk_info[u].ndblks;
    }

    
    hdr->stats.computed.hdr_size = hdr->size = H5EA_HEADER_SIZE_HDR(hdr);

    
    if (hdr->cparam.cls->crt_context) {
        if (NULL == (hdr->cb_ctx = (*hdr->cparam.cls->crt_context)(ctx_udata)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, FAIL,
                        "unable to create extensible array client callback context");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5EA__hdr_alloc_elmts(H5EA_hdr_t *hdr, size_t nelmts)
{
    void    *elmts = NULL; 
    unsigned idx;          
    void    *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(nelmts > 0);

    
    H5_CHECK_OVERFLOW(nelmts,  size_t,  uint32_t);
    idx = H5VM_log2_of2((uint32_t)nelmts) - H5VM_log2_of2((uint32_t)hdr->cparam.data_blk_min_elmts);

    
    if (idx >= hdr->elmt_fac.nalloc) {
        H5FL_fac_head_t **new_fac; 
        size_t            new_nalloc =
            MAX3(1, (idx + 1), (2 * hdr->elmt_fac.nalloc)); 

        
        if (NULL == (new_fac = H5FL_SEQ_REALLOC(H5FL_fac_head_ptr_t, hdr->elmt_fac.fac, new_nalloc)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for data block data element buffer factory array");

        
        memset(new_fac + hdr->elmt_fac.nalloc, 0,
               (new_nalloc - hdr->elmt_fac.nalloc) * sizeof(H5FL_fac_head_ptr_t));

        
        hdr->elmt_fac.nalloc = new_nalloc;
        hdr->elmt_fac.fac    = new_fac;
    } 

    
    if (NULL == hdr->elmt_fac.fac[idx]) {
        if (NULL == (hdr->elmt_fac.fac[idx] = H5FL_fac_init(nelmts * (size_t)hdr->cparam.cls->nat_elmt_size)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTINIT, NULL,
                        "can't create data block data element buffer factory");
    } 

    
    if (NULL == (elmts = H5FL_FAC_MALLOC(hdr->elmt_fac.fac[idx])))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for data block data element buffer");

    
    ret_value = elmts;

done:
    if (!ret_value)
        if (elmts)
            elmts = H5FL_FAC_FREE(hdr->elmt_fac.fac[idx], elmts);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_free_elmts(H5EA_hdr_t *hdr, size_t nelmts, void *elmts)
{
    unsigned idx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(nelmts > 0);
    assert(elmts);

    
    H5_CHECK_OVERFLOW(nelmts,  size_t,  uint32_t);
    idx = H5VM_log2_of2((uint32_t)nelmts) - H5VM_log2_of2((uint32_t)hdr->cparam.data_blk_min_elmts);

    
    assert(idx < hdr->elmt_fac.nalloc);
    assert(hdr->elmt_fac.fac[idx]);
    elmts = H5FL_FAC_FREE(hdr->elmt_fac.fac[idx], elmts);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

haddr_t
H5EA__hdr_create(H5F_t *f, const H5EA_create_t *cparam, void *ctx_udata)
{
    H5EA_hdr_t *hdr       = NULL;  
    bool        inserted  = false; 
    haddr_t     ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cparam);

#ifndef NDEBUG
    {
        unsigned sblk_idx;         
        size_t   dblk_nelmts;      
        size_t   dblk_page_nelmts; 

        
        if (cparam->raw_elmt_size == 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF, "element size must be greater than zero");
        if (cparam->max_nelmts_bits == 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "max. # of elements bits must be greater than zero");
        if (cparam->max_nelmts_bits > H5EA_MAX_NELMTS_IDX_MAX)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF, "max. # of elements bits must be <= %u",
                        (unsigned)H5EA_MAX_NELMTS_IDX_MAX);
        if (cparam->sup_blk_min_data_ptrs < 2)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "min # of data block pointers in super block must be >= two");
        if (!POWER_OF_TWO(cparam->sup_blk_min_data_ptrs))
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "min # of data block pointers in super block must be power of two");
        if (!POWER_OF_TWO(cparam->data_blk_min_elmts))
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "min # of elements per data block must be power of two");
        dblk_page_nelmts = (size_t)1 << cparam->max_dblk_page_nelmts_bits;
        if (dblk_page_nelmts < cparam->idx_blk_elmts)
            HGOTO_ERROR(
                H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                "# of elements per data block page must be greater than # of elements in index block");

        
        sblk_idx    = H5EA_SBLK_FIRST_IDX(cparam->sup_blk_min_data_ptrs);
        dblk_nelmts = H5EA_SBLK_DBLK_NELMTS(sblk_idx, cparam->data_blk_min_elmts);
        if (dblk_page_nelmts < dblk_nelmts)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "max. # of elements per data block page bits must be > # of elements in "
                        "first data block from super block");

        if (cparam->max_dblk_page_nelmts_bits > cparam->max_nelmts_bits)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "max. # of elements per data block page bits must be <= max. # of elements bits");
    }
#endif 

    
    if (NULL == (hdr = H5EA__hdr_alloc(f)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "memory allocation failed for extensible array shared header");

    
    hdr->idx_blk_addr = HADDR_UNDEF;

    
    H5MM_memcpy(&hdr->cparam, cparam, sizeof(hdr->cparam));

    
    if (H5EA__hdr_init(hdr, ctx_udata) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINIT, HADDR_UNDEF,
                    "initialization failed for extensible array header");

    
    if (HADDR_UNDEF == (hdr->addr = H5MF_alloc(f, H5FD_MEM_EARRAY_HDR, (hsize_t)hdr->size)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "file allocation failed for extensible array header");

    
    if (hdr->swmr_write)
        if (NULL == (hdr->top_proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, HADDR_UNDEF, "can't create extensible array entry proxy");

    
    if (H5AC_insert_entry(f, H5AC_EARRAY_HDR, hdr->addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINSERT, HADDR_UNDEF, "can't add extensible array header to cache");
    inserted = true;

    
    if (hdr->top_proxy)
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, f, hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add extensible array entry as child of array proxy");

    
    ret_value = hdr->addr;

done:
    if (!H5_addr_defined(ret_value))
        if (hdr) {
            
            if (inserted)
                if (H5AC_remove_entry(hdr) < 0)
                    HDONE_ERROR(H5E_EARRAY, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove extensible array header from cache");

            
            if (H5_addr_defined(hdr->addr) &&
                H5MF_xfree(f, H5FD_MEM_EARRAY_HDR, hdr->addr, (hsize_t)hdr->size) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF, "unable to free extensible array header");

            
            if (H5EA__hdr_dest(hdr) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to destroy extensible array header");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_incr(H5EA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->rc == 0)
        if (H5AC_pin_protected_entry(hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTPIN, FAIL, "unable to pin extensible array header");

    
    hdr->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_decr(H5EA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->rc);

    
    hdr->rc--;

    
    if (hdr->rc == 0) {
        assert(hdr->file_rc == 0);
        if (H5AC_unpin_entry(hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPIN, FAIL, "unable to unpin extensible array header");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_fuse_incr(H5EA_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);

    
    hdr->file_rc++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

size_t
H5EA__hdr_fuse_decr(H5EA_hdr_t *hdr)
{
    size_t ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(hdr->file_rc);

    
    hdr->file_rc--;

    
    ret_value = hdr->file_rc;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_modified(H5EA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->f);

    
    if (H5AC_mark_entry_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTMARKDIRTY, FAIL, "unable to mark extensible array header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5EA_hdr_t *
H5EA__hdr_protect(H5F_t *f, haddr_t ea_addr, void *ctx_udata, unsigned flags)
{
    H5EA_hdr_t         *hdr;   
    H5EA_hdr_cache_ud_t udata; 
    H5EA_hdr_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(ea_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.f         = f;
    udata.addr      = ea_addr;
    udata.ctx_udata = ctx_udata;

    
    if (NULL == (hdr = (H5EA_hdr_t *)H5AC_protect(f, H5AC_EARRAY_HDR, ea_addr, &udata, flags)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect extensible array header, address = %llu", (unsigned long long)ea_addr);
    hdr->f = f; 

    
    if (hdr->swmr_write && NULL == hdr->top_proxy) {
        
        if (NULL == (hdr->top_proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTCREATE, NULL, "can't create extensible array entry proxy");

        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, f, hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, NULL,
                        "unable to add extensible array entry as child of array proxy");
    } 

    
    ret_value = hdr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_unprotect(H5EA_hdr_t *hdr, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5AC_unprotect(hdr->f, H5AC_EARRAY_HDR, hdr->addr, hdr, cache_flags) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect extensible array hdr, address = %llu",
                    (unsigned long long)hdr->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_delete(H5EA_hdr_t *hdr)
{
    unsigned cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t   ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(!hdr->file_rc);

#ifndef NDEBUG
    unsigned hdr_status = 0; 

    
    if (H5AC_get_entry_status(hdr->f, hdr->addr, &hdr_status) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "unable to check metadata cache status for array header");

    
    assert(hdr_status & H5AC_ES__IN_CACHE);
    assert(hdr_status & H5AC_ES__IS_PROTECTED);
#endif 

    
    if (H5_addr_defined(hdr->idx_blk_addr)) {
        
        if (H5EA__iblock_delete(hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDELETE, FAIL, "unable to delete extensible array index block");
    } 

    
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

done:

    
    if (H5EA__hdr_unprotect(hdr, cache_flags) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__hdr_dest(H5EA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->rc == 0);

    
    if (hdr->cb_ctx) {
        if ((*hdr->cparam.cls->dst_context)(hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL,
                        "unable to destroy extensible array client callback context");
    } 
    hdr->cb_ctx = NULL;

    
    if (hdr->elmt_fac.fac) {
        unsigned u; 

        
        assert(hdr->elmt_fac.nalloc > 0);

        
        for (u = 0; u < hdr->elmt_fac.nalloc; u++) {
            
            if (hdr->elmt_fac.fac[u]) {
                if (H5FL_fac_term(hdr->elmt_fac.fac[u]) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL,
                                "unable to destroy extensible array header factory");
                hdr->elmt_fac.fac[u] = NULL;
            } 
        }     

        
        hdr->elmt_fac.fac = (H5FL_fac_head_t **)H5FL_SEQ_FREE(H5FL_fac_head_ptr_t, hdr->elmt_fac.fac);
    } 

    
    if (hdr->sblk_info)
        hdr->sblk_info = (H5EA_sblk_info_t *)H5FL_SEQ_FREE(H5EA_sblk_info_t, hdr->sblk_info);

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_dest(hdr->top_proxy) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL, "unable to destroy extensible array 'top' proxy");
        hdr->top_proxy = NULL;
    } 

    
    hdr = H5FL_FREE(H5EA_hdr_t, hdr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
