#' @title Bias Corrected Meta-Analysis with Dirichlet Mixture Process Priors for the biased component
#'
#' @description This function performers a Bayesian meta-analysis with DPM as random effects
#'
#' @param data                A data frame with at least two columns with the following names:
#'                             1) TE = treatment effect,
#'                             2) seTE = the standard error of the treatment effect.
#'
#' @param mean.mu.theta       Prior mean of the mean of the base distribution default value is mean.mu.theta = 0.
#'
#' @param sd.mu.theta         Prior standard deviation of the base distribution, the default value is 10^-6.
#'
#' @param scale.sigma.between Prior scale parameter for scale gamma distribution for the
#'                            precision between studies. The default value is 0.5.
#'
#' @param df.scale.between    Degrees of freedom of the scale gamma distribution for the precision between studies.
#'                            The default value is 1, which results in a Half Cauchy distribution for the standard
#'                            deviation between studies. Larger values e.g. 30 corresponds to a Half Normal distribution.
#' @param scale.sigma.beta    Prior scale parameter for the scale.gamma distribution for the
#'                            precision between study biases.
#'
#' @param df.scale.beta      Degrees of freedom of the scale gamma distribution for the precision between
#'                            study biases. The default value is 1, which results in a Half Cauchy distribution
#'                            for the standard deviation between biases.
#'
#' @param B.lower             Lower bound of the bias parameter B, the default value is -15.
#' @param B.upper             Upper bound of the bias parameter B, the default value is 15.
#'
#' @param a.0                 Parameter for the prior Beta distribution for the probability of bias. Default value is a0 = 0.5.
#' @param a.1                 Parameter for the prior Beta distribution for the probability of bias. Default value is a1 = 1.
#'
#'
#' @param alpha.0             Lower bound of the uniform prior for the concentration parameter for the DP,
#'                            the default value is 0.5.
#' @param alpha.1             Upper bound of the uniform prior for the concentration parameter for the DP,
#'                            the default value depends on the sample size, see the example below. We give as
#'                            working value alpha.1 = 2
#'
#' @param K                   Maximum number of clusters in the DP, the default value depends on alpha.1, see the
#'                            example below. We give as working value K = 10.
#'
#' @param nr.chains           Number of chains for the MCMC computations, default 2.
#' @param nr.iterations       Number of iterations after adapting the MCMC, default is 10000. Some models may need more iterations.
#' @param nr.adapt            Number of iterations in the adaptation process, default is 1000. Some models may need more iterations during adptation.
#' @param nr.burnin           Number of iteration discard for burn-in period, default is 1000. Some models may need a longer burnin period.
#' @param nr.thin             Thinning rate, it must be a positive integer, the default value 1.
#' @param parallel            NULL -> jags, 'jags.parallel' -> jags.parallel execution
#'
#' @return                    This function returns an object of the class "bcmixmeta". This object contains the MCMC
#'                            output of each parameter and hyper-parameter in the model and
#'                            the data frame used for fitting the model.
#'
#'
#' @details The results of the object of the class bcmixmeta can be extracted with R2jags or with rjags. In addition a summary, a print and a plot functions are
#' implemented for this type of object.
#'
#'
#' @references Verde, P.E., and Rosner, G.L. (2025), A Bias-Corrected Bayesian Nonparametric Model for Combining Studies With Varying Quality in Meta-Analysis. Biometrical Journal., 67: e70034. https://doi.org/10.1002/bimj.70034
#'
#'
#' @examples
#' \dontrun{
#' library(jarbes)
#'
#'
#' # Example: Stemcells
#'
#' data("stemcells")
#' stemcells$TE = stemcells$effect.size
#' stemcells$seTE = stemcells$se.effect
#'
#' # Beta(0.5, 1)
#' a.0 = 0.5
#' a.1 = 1
#'
#' # alpha.max
#'  N = dim(stemcells)[1]
#'  alpha.max = 1/5 *((N-1)*a.0 - a.1)/(a.0 + a.1)
#'
#' alpha.max
#'
#'# K.max
#' K.max = 1 + 5*alpha.max
#' K.max = round(K.max)
#'
#' K.max
#'
#' set.seed(20233)
#'
#' bcmix.2.stemcell = bcmixmeta(stemcells,
#'                             mean.mu.theta=0, sd.mu.theta=100,
#'                             B.lower = -15,
#'                             B.upper = 15,
#'                             alpha.0 = 0.5,
#'                             alpha.1 = alpha.max,
#'                             a.0 = a.0,
#'                             a.1 = a.1,
#'                             K = K.max,
#'                             df.scale.between = 1,
#'                             scale.sigma.between = 0.5,
#'                             nr.chains = 4,
#'                             nr.iterations = 50000,
#'                             nr.adapt = 1000,
#'                             nr.burnin = 10000,
#'                             nr.thin = 4)
#'
#'
#'  diagnostic(bcmix.2.stemcell, y.lim = c(-1, 15), title.plot = "Default priors")
#'
#'
#'  bcmix.2.stemcell.mcmc <- as.mcmc(bcmix.1.stemcell$BUGSoutput$sims.matrix)
#'
#'
#'theta.names <- paste(paste("theta[",1:31, sep=""),"]", sep="")
#'theta.b.names <- paste(paste("theta.bias[",1:31, sep=""),"]", sep="")
#'
# Greeks ...
#'theta.b.greek.names <- paste(paste("theta[",1:31, sep=""),"]^B", sep="")
#'theta.greek.names <- paste(paste("theta[",1:31, sep=""),"]", sep="")
#'
#'
#'caterplot(bcmix.2.stemcell.mcmc,
#'          parms = theta.names,               # theta
#'          labels = theta.greek.names,
#'          greek = T,
#'          labels.loc="axis", cex =0.7,
#'          col = "black",
#'          style = "plain",
#'          reorder = F,
#'          val.lim =c(-6, 16),
#'          quantiles = list(outer=c(0.05,0.95),inner=c(0.16,0.84)),
#'          x.lab = "Effect: mean difference"
#')
#'title( "95% posterior intervals of studies' effects")
#'caterplot(bcmix.2.stemcell.mcmc,
#'          parms = theta.b.names,             # theta.bias
#'          labels = theta.greek.names,
#'          greek = T,
#'          labels.loc="no",
#'          cex = 0.7,
#'          col = "grey",
#'          style = "plain", reorder = F,
#'          val.lim =c(-6, 16),
#'          quantiles = list(outer=c(0.025,0.975),inner=c(0.16,0.84)),
#'          add = TRUE,
#'          collapse=TRUE, cat.shift= -0.5,
#')
#'
#'attach.jags(bcmix.2.stemcell, overwrite = TRUE)
#'abline(v=mean(mu.theta), lwd =2, lty =2)
#'
#'legend(9, 20, legend = c("bias corrected", "biased"),
#'      lty = c(1,1), lwd = c(2,2), col = c("black", "grey"))
#'
#'
#' }
#'
#' @import R2jags
#' @import rjags
#'
#'
#' @export
bcmixmeta = function(
    data,
    # Hyperpriors parameters............................................
    # Hyperpriors parameters............................................
    mean.mu.theta     = 0,
    sd.mu.theta       = 10,
    scale.sigma.between = 0.5,
    df.scale.between    = 1,
    scale.sigma.beta    = 0.5,
    df.scale.beta       = 1,
    B.lower             = -15,
    B.upper             = 15,
    a.0                 = 0.5,
    a.1                 = 1,
    alpha.0             = 0.03,
    alpha.1             = 2,
    K                   = 10,
    # MCMC setup..........................................................
    nr.chains       = 2,
    nr.iterations   = 10000,
    nr.adapt        = 1000,
    nr.burnin       = 1000,
    nr.thin         = 1,
    parallel        = NULL)UseMethod("bcmixmeta")


#' @export
bcmixmeta.default = function(
    data,
    # Hyperpriors parameters............................................
    mean.mu.theta           = 0,
    sd.mu.theta             = 10,
    scale.sigma.between = 0.5,
    df.scale.between    = 1,
    scale.sigma.beta    = 0.5,
    df.scale.beta       = 1,
    B.lower             = -15,
    B.upper             = 15,
    a.0                 = 0.5,
    a.1                 = 1,
    alpha.0             = 0.03,
    alpha.1             = 2,
    K                   = 10,
    # MCMC setup........................................................
    nr.chains       = 2,
    nr.iterations   = 10000,
    nr.adapt        = 1000,
    nr.burnin       = 1000,
    nr.thin         = 1,
    parallel        = NULL
)
{
  if(!is.null(parallel) && parallel != "jags.parallel") stop("The parallel option must be NULL or 'jags.parallel'")


  # Avoid order of the effects and we add NA for posterior prediction
  y = c(data$TE, NA)
  se.y = c(data$seTE, 1)
  N = length(y)

  # I with order information ...
  I = rep(NA, N)
  min.index = which(y == min(y, na.rm = TRUE))
  max.index = which(y == max(y, na.rm = TRUE))
  #I[min.index] = 1                                # This is the case of both extremes as biased studies => I get similar to the RE, but heavy tail
  I[max.index] = 1


  if(N<5)stop("Low number of studies in the meta-analysis!")

  # This list describes the data used by the BUGS script.
  data.bcmixmeta =
    list(y = y,
         se.y = se.y,
         N = N,
         I = I,
         mean.mu.theta = mean.mu.theta,
         sd.mu.theta = sd.mu.theta,
         scale.sigma.between = scale.sigma.between,
         df.scale.between = df.scale.between,
         scale.sigma.beta = scale.sigma.beta,
         df.scale.beta = df.scale.beta,
         B.lower = B.lower,
         B.upper = B.upper,
         a.0 = a.0,
         a.1 = a.1,
         alpha.0 = alpha.0,
         alpha.1 = alpha.1,
         K = K
    )

  # List of parameters
  par.bcmixmeta  <- c("mu.k",
                      "mu.theta",
                      "sd.theta",
                      "pi",
                      "alpha",
                      "pi.B",
                      "mu.beta",
                      "sd.beta",
                      "K.hat",
                      "theta.bias",
                      "theta",
                      "beta",
                       "I",
                      "group",
                      "gind",
                      "equalsmatrix.bias")

  # Model in BUGS with mu_beta ....
  model.bugs.bcmixmeta =
    "
  model
  {
   for( i in 1 : N ) {
  # Likelihood of theta.bias[i] ...................................................
            y[i] ~ dnorm(theta.bias[i], 1 / se.y[i]^2)

  # Mixture Process: Normal and Dirichlet process ..............................

  theta.bias[i] <- theta[i] + (beta[i]*I[i])
           I[i] ~ dbern(pi.B)
       theta[i] ~ dnorm(mu.theta, inv.var.theta)

  # Dirichlet Process for biased component .....................................
           beta[i] <- mu.k[group[i]]
           group[i] ~ dcat(pi[])

  for(j in 1:K)
  {gind[i, j] <- equals(j, group[i])}   # Frequency of study i belong to cluster j

   }

  # Number of studies in the same bias cluster
  for(i in 1:N)
  {new.group[i] <- (1-I[i]) + I[i]*(group[i]+1)}

  for(i in 1:N){
    for(j in 1:N){
    equalsmatrix.bias[i,j] <- equals(new.group[i], new.group[j])
    }
  }

  # Counting the number of clusters ...
  K.hat <- sum(cl[])
  for(j in 1:K)
  {
  sumind[j] <- sum(gind[,j])
      cl[j] <- step(sumind[j] -1 + 0.01) # cluster j used in this iteration.
  }


  # Prior for the probability of bias.........................................
     pi.B ~ dbeta(a.0, a.1)

  # Priors for model of interest ..............................................

          mu.theta ~ dnorm(mean.mu.theta, inv.var.mu.theta)
          inv.var.mu.theta <- pow(sd.mu.theta, -2)

     inv.var.theta ~ dscaled.gamma(scale.sigma.between, df.scale.between)
          sd.theta <- pow(inv.var.theta, -0.5)

  # Priors for the clusters' parameters biased component .......................

         mu.beta ~ dunif(B.lower, B.upper)            # mean of the base distribution in G_beta

   for(k in 1:K){
        mu.k[k] ~ dnorm(mu.beta, inv.var.beta)       # The mean of the cluster k in the DP
        #mu.k[k] ~ dnorm(0, inv.var.beta)
        #mu.k[k] ~ dnorm(0, 0.1)                       # the scale is very important
       }

  inv.var.beta ~ dscaled.gamma(scale.sigma.beta, df.scale.beta)
  sd.beta = pow(inv.var.beta, -0.5)

 # Prior for alpha .............................................................
  alpha ~ dunif(alpha.0, alpha.1)

 # Stick-Breaking process and sampling from G_beta..............................

      q[1] ~ dbeta(1, alpha)
      p[1] <- q[1]
      for (k in 2:K) {
        q[k] ~ dbeta(1, alpha)
        p[k] <- q[k] * prod(1 - q[1:(k-1)])
      }

      # Normalized cluster probs
      for (k in 1:K) {
        pi[k] <- p[k] / sum(p[1:K])
      }

  }
  "

  model.bugs.connection = textConnection(model.bugs.bcmixmeta)
  model.bugs = model.bugs.bcmixmeta                  # To save the string of the BUGS model


  # Use R2jags as interface for JAGS ...

  if (is.null(parallel)) { #execute R2jags
    model.bugs.connection <- textConnection(model.bugs)

    # Use R2jags as interface for JAGS ...

    results <- jags( data = data.bcmixmeta,
                     parameters.to.save = par.bcmixmeta,
                     model.file = model.bugs.connection,
                     n.chains = nr.chains,
                     n.iter = nr.iterations,
                     n.burnin = nr.burnin,
                     n.thin = nr.thin,
                     DIC = TRUE,
                     pD=TRUE)

    # Close text connection
    close(model.bugs.connection)
  }else if(parallel == "jags.parallel"){
    writeLines(model.bugs, "model.bugs")
    results <- jags.parallel(     data = data.bcmixmeta,
                                  parameters.to.save = par.bcmixmeta,
                                  model.file = "model.bugs",
                                  n.chains = nr.chains,
                                  n.iter = nr.iterations,
                                  n.burnin = nr.burnin,
                                  n.thin = nr.thin,
                                  DIC=TRUE)

    # Compute pD from result
    results$BUGSoutput$pD = results$BUGSoutput$DIC - results$BUGSoutput$mean$deviance

    # Delete model.bugs on exit ...
    unlink("model.bugs")
  }


  # Extra outputs that are linked with other functions ...
  results$data = data

  # BUGS code .......................................................
  results$model.bugs                = model.bugs                           #?
  # Hyperpriors parameters............................................
  results$prior$mean.mu             = mean.mu.theta
  results$prior$sd.mu               = sd.mu.theta
  results$prior$scale.sigma.between = scale.sigma.between
  results$prior$df.scale.between    = df.scale.between
  results$prior$scale.sigma.beta    = scale.sigma.beta  # <-- ADD THIS LINE
  results$prior$df.scale.beta       = df.scale.beta     # <-- ADD THIS LINE
  results$prior$B.lower             = B.lower
  results$prior$B.upper             = B.upper
  results$prior$a.0                 = a.0
  results$prior$a.1                 = a.1
  results$prior$alpha.0             = alpha.0
  results$prior$alpha.1             = alpha.1
  results$prior$K                   = K
  results$N

  class(results) = c("bcmixmeta")

  return(results)
}


#' Generic print function for bcmixmeta object in jarbes.
#'
#' @param x The object generated by the function bcmixmeta.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#'
#' @param ... \dots
#'
#' @export
print.bcmixmeta <- function(x, digits, ...)
{
  print(x$BUGSoutput,...)
}



