% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{mirai_map}
\alias{mirai_map}
\title{mirai Map}
\usage{
mirai_map(.x, .f, ..., .args = list(), .promise = NULL, .compute = NULL)
}
\arguments{
\item{.x}{(list | vector | matrix | data.frame) input to map over. For matrix
or dataframe, maps over rows (see Multiple Map section).}

\item{.f}{(function) applied to each element of \code{.x}, or each row of a
matrix / dataframe.}

\item{...}{(named arguments) objects referenced but not defined in \code{.f}.}

\item{.args}{(list) constant arguments passed to \code{.f}.}

\item{.promise}{(function | list) registers a promise against each mirai.
Either an \code{onFulfilled} function, or a list of (\code{onFulfilled},
\code{onRejected}) functions for \code{\link[promises:then]{promises::then()}}. Requires the \pkg{promises}
package.}

\item{.compute}{(character) name of the compute profile. Each profile has its
own independent set of daemons. \code{NULL} (default) uses the 'default'
profile.}
}
\value{
A 'mirai_map' (list of 'mirai' objects).
}
\description{
Asynchronous parallel map of a function over a list or vector using
\pkg{mirai}, with optional \pkg{promises} integration. For matrix or
dataframe inputs, maps over rows.
}
\details{
Sends each application of function \code{.f} on an element of \code{.x} (or row of
\code{.x}) for computation in a separate \code{\link[=mirai]{mirai()}} call. If \code{.x} is named, names
are preserved.

Takes advantage of \pkg{mirai} scheduling to minimise overall execution
time.

Facilitates recovery from partial failure by returning all 'miraiError' /
'errorValue' as the case may be, thus allowing only failures to be re-run.

This function requires daemons to have previously been set, and will error
otherwise.
}
\section{Collection Options}{


\code{x[]} collects the results of a 'mirai_map' \code{x} and returns a list. This will
wait for all asynchronous operations to complete if still in progress,
blocking but user-interruptible.

\code{x[.flat]} collects and flattens map results to a vector, checking that
they are of the same type to avoid coercion. Note: errors if an 'errorValue'
has been returned or results are of differing type.

\code{x[.progress]} collects map results whilst showing a progress bar from
the \pkg{cli} package, if installed, with completion percentage and ETA, or
else a simple text progress indicator. Note: if the map operation completes
too quickly then the progress bar may not show at all.

\code{x[.stop]} collects map results applying early stopping, which stops at
the first failure and cancels remaining operations.

The options above may be combined in the manner of: \cr
\code{x[.stop, .progress]} which applies early stopping together with a
progress indicator.
}

\section{Multiple Map}{


If \code{.x} is a matrix or dataframe (or other object with 'dim' attributes),
\emph{multiple} map is performed over its \strong{rows}. Character row names are
preserved as names of the output.

This allows map over 2 or more arguments, and \code{.f} should accept at least as
many arguments as there are columns. If the dataframe has column names, or
the matrix has column dimnames, arguments are passed to \code{.f} by name.

To map over \strong{columns} instead, first wrap a dataframe in \code{\link[=as.list]{as.list()}}, or
transpose a matrix using \code{\link[=t]{t()}}.
}

\section{Nested Maps}{


To run maps within maps, the function provided to the outer map must include
a call to \code{\link[=daemons]{daemons()}} to set daemons for the inner map. To guard against
inadvertently spawning an excessive number of daemons on the same machine,
attempting to launch local daemons within a map using \code{daemons(n)} will
error.

When the outer daemons run on remote machines and you want local daemons on
each, use 2 separate calls instead of \code{daemons(n)}:
\verb{daemons(url = local_url()); launch_local(n)}. This is equivalent, and is
permitted from within a map.
}

\examples{
\dontshow{if (interactive()) withAutoprint(\{ # examplesIf}
daemons(4)

# perform and collect mirai map
mm <- mirai_map(c(a = 1, b = 2, c = 3), rnorm)
mm
mm[]

# map with constant args specified via '.args'
mirai_map(1:3, rnorm, .args = list(n = 5, sd = 2))[]

# flatmap with helper function passed via '...'
mirai_map(
  10^(0:9),
  function(x) rnorm(1L, valid(x)),
  valid = function(x) min(max(x, 0L), 100L)
)[.flat]

# unnamed matrix multiple map: arguments passed to function by position
(mat <- matrix(1:4, nrow = 2L))
mirai_map(mat, function(x = 10, y = 0, z = 0) x + y + z)[.flat]

# named matrix multiple map: arguments passed to function by name
(mat <- matrix(1:4, nrow = 2L, dimnames = list(c("a", "b"), c("y", "z"))))
mirai_map(mat, function(x = 10, y = 0, z = 0) x + y + z)[.flat]

# dataframe multiple map: using a function taking '...' arguments
df <- data.frame(a = c("Aa", "Bb"), b = c(1L, 4L))
mirai_map(df, function(...) sprintf("\%s: \%d", ...))[.flat]

# indexed map over a vector (using a dataframe)
v <- c("egg", "got", "ten", "nap", "pie")
mirai_map(
  data.frame(1:length(v), v),
  sprintf,
  .args = list(fmt = "\%d_\%s")
)[.flat]

# return a 'mirai_map' object, check for resolution, collect later
mp <- mirai_map(2:4, function(x) runif(1L, x, x + 1))
unresolved(mp)
mp
mp[.flat]
unresolved(mp)

# progress indicator counts up from 0 to 4 seconds
res <- mirai_map(1:4, Sys.sleep)[.progress]

# stops early when second element returns an error
tryCatch(mirai_map(list(1, "a", 3), sum)[.stop], error = identity)

daemons(0)
\dontshow{\}) # examplesIf}
\dontshow{if (interactive() && requireNamespace("promises", quietly = TRUE)) withAutoprint(\{ # examplesIf}
# promises example that outputs the results, including errors, to the console
daemons(1, dispatcher = FALSE)
ml <- mirai_map(
  1:30,
  function(i) {Sys.sleep(0.1); if (i == 30) stop(i) else i},
  .promise = list(
    function(x) cat(paste(x, "")),
    function(x) { cat(conditionMessage(x), "\n"); daemons(0) }
  )
)
\dontshow{\}) # examplesIf}
}
