# Code to fit symmetry, quasi-symmetry and marginal homogeneity models that satisfy
# minimum discriminant information criterion.
#
#' Fits symmetry model.
#'
#' Ireland, C. T., Ku, H. H., & Kullback, S. (1969).  Symmetry and marginal homogeneity of
#' an r × r contingency table.  Journal of the American Statistical Association, 64(328),
#' 1323-1341.
#' @param n matrix of observed counts
#' @param truncated should the diagonal be excluded.  Default is FALSE, include the diagonal.
#' @returns a list containing
#'    mdis: value of the minimum discriminant information statistic (appox chi-squared)
#'    df: dgrees of freedom
#'    x_star: matrix of model-based counts
#'    p_star: matrix of model-based p-values
#' @export
#' @examples
#' Ireland_symmetry(vision_data)
Ireland_symmetry <- function(n, truncated=FALSE) {
  r <- nrow(n)
  if (truncated) {
    n <- Ireland_normalize_for_truncation(n)
  }
  N <- sum(n)
  pi <- n / N
  d <- 0.0
  for (i in 1:r) {
    for (j in 1:r) {
      if (truncated && i == j) {
        next
      }
      d <- d + (sqrt(pi[i, j] * pi[j, i]))
    }
  }

  p_star <- matrix(0.0, nrow=r, ncol=r)
  x_star <- matrix(0.0, nrow=r, ncol=r)
  d <- 1.0 / d
  for (i in 1:r) {
    for (j in 1:r) {
      if (truncated && i == j) {
        next
      }
      p_star[i, j] <- d * sqrt(pi[i, j] * pi[j, i])
      x_star[i, j] <- N * p_star[i, j]
    }
  }
  mdis <- Ireland_mdis(n, x_star, truncated)
  df <- r * (r - 1) / 2
  list(mdis=mdis, df=df, x_star=x_star, p_star=p_star)
}


#' Fits marginal homogeneity model
#'
#' Fits the marginal homogeneity model according to the minimum discriminant information.
#' Ireland, C. T., Ku, H. H., & Kullback, S. (1969).  Symmetry and marginal homogeneity of
#' an r × r contingency table.  Journal of the American Statistical Association, 64(328),
#' 1323-1341.
#' @param n matrix of observed counts
#' @param truncated should the diagonal be excluded.  Default is FALSE, include the diagonal.
#' @param max_iter maximum number of iterations to perform
#' @param verbose should cycle-by-cycle information be printed out. Default is FALSE.
#' @returns a list containing
#'    mdis: value of the minimum discriminant information statistic (appox chi-squared)
#'    df: dgrees of freedom
#'    x_star: matrix of model-based counts
#'    p_star: matrix of model-based p-values
#' @export
#' @examples
#' Ireland_marginal_homogeneity(vision_data)
Ireland_marginal_homogeneity <- function(n, truncated=FALSE, max_iter=15,
                                         verbose=FALSE) {
  r <- nrow(n)
  if (truncated) {
    n <- Ireland_normalize_for_truncation(n)
  }
  N <- sum(n)

  p_star <- matrix(0.0, nrow=r, ncol=r)
  p_star <- n / N
  for (iter in 1:max_iter) {
    p_i_dot <- rowSums(p_star)
    p_dot_j <- colSums(p_star)
    c_n <- 0.0
    for (i in 1:r) {
      for (j in 1:r) {
        if (truncated && i == j) {
          next
        }
        ratio <- sqrt(p_dot_j[i] * p_i_dot[j] / (p_i_dot[i] * p_dot_j[j])) * p_star[i, j]
        c_n <- c_n + ratio
      }
    }
    c_n <- 1.0 / c_n

    diff <- 0.0
    for (i in 1:r) {
      for (j in 1:r) {
        if (truncated && i == j) {
          next
        }

        p_star_new <- sqrt(p_dot_j[i] * p_i_dot[j] / (p_i_dot[i] * p_dot_j[j])) * p_star[i, j] * c_n
        diff <- max(abs(p_star[i, j] - p_star_new), diff)
        p_star[i, j] <- p_star_new
      }
    }
    if (verbose) {
      message(paste(iter, diff))
    }
  }

  x_star <- N * p_star
  mdis <- Ireland_mdis(n, x_star, truncated)
  df <- r - 1
  list(mdis=mdis, df=df, x_star=x_star, p_star=p_star)
}


#' Computes the MDIS between the two matrices provided.
#'
#' @param n first matrix (usually observed counts)
#' @param x_star second matrix (usually model-based)
#' @param truncated should the diagonal be ignored. Default is FALSE, include
#' the diagonal elements.
#' @returns value of the MDIS criterion
Ireland_mdis <- function(n, x_star, truncated=FALSE) {
  epsilon = 1.0e-8
  r <- nrow(n)
  N <- sum(n)
  mdis <- 0.0
  for (i in 1:r) {
    for (j in 1:r) {
      if (truncated && i == j) {
        next
      }
      if (n[i, j] < epsilon) {
        n[i, j] <- epsilon
      }
      if (x_star[i, j] < epsilon) {
        x_star[i, j] <- epsilon
      }
      mdis <- mdis + x_star[i, j] * log(x_star[i, j] / n[i, j])
    }
  }
  mdis <- 2.0 * mdis
  mdis
}


#' Fit for quasi-symmetry model. Obtained by subtraction, so no model-based probabilities.
#'
#' @param n matrix of observed counts
#' @param truncated should the diagonal be excluded, Default is FALSE, include the diagonal.
#' @returns a list with mdis = MDIS value and df = degrees of freedom for quasi-symmetry model
#' @seealso [Ireland_quasi_symmetry_model()]
#' @export
#' @examples
#' Ireland_quasi_symmetry(vision_data)
Ireland_quasi_symmetry <- function(n, truncated=FALSE) {
  r <- nrow(n)
  result_symmetry <- Ireland_symmetry(n, truncated)
  result_homogeneity <- Ireland_marginal_homogeneity(n, truncated)
  mdis <- result_symmetry$mdis - result_homogeneity$mdis
  df <- (r - 1) * (r - 2) / 2
  list(mdis=mdis, df=df)
}


#' Renormalize counts to account for truncation of diagonal
#'
#' @param n matrix of observed counts
#' @returns matrix n with diagonal set to 0.0
Ireland_normalize_for_truncation <- function(n) {
  r <- ncol(n)
  for (i in 1:r) {
    n[i, i] <- 0.0
  }
  n
}


#' Fitss the quasi-symmetry model.
#'
#' Fits the model according to the MDIS criterion.
#' @param n matrix of observed counts
#' @param truncated should the diagonal be excluded. Default is FALSE, include diagonal cells.
#' @param max_iter maximum number of iterations in minimizing the criterion.  Default is 4
#' @param verbose logical variable, should cycle-by-cycle info be printed. Defaullt is FALSE.
#' @returns a list containing
#'    mdis: value of the MDIS at termination
#'    df: degrees of freedom
#'    x_star: matrix of model-reproduced counts
#'    p_star: matrix of model-reproduced p-values
#' @seealso [Ireland_quasi_symmetry()]
#' @export
#' @examples
#' Ireland_quasi_symmetry_model(vision_data)
Ireland_quasi_symmetry_model <- function(n, truncated=FALSE, max_iter=5,
                                         verbose=FALSE) {
  epsilon <- 1.0e-8
  r <- nrow(n)
  if (truncated) {
    n <- Ireland_normalize_for_truncation(n)
  }
  N <- sum(n)
  p <- n / N

  t_ij <- t(p) + p
  p_i_dot <- rowSums(p)
  p_dot_j <- colSums(p)

  result <- Ireland_symmetry(n, truncated)
  p_star <- result$p_star
  for (i in 1:r) {
    for (j in 1:r) {
      if (p_star[i, j] < epsilon) {
        p_star[i, j] <- epsilon
      }
    }
  }
  for (iter in 1:max_iter) {
    initial <- p_star
    # step 3n
    t_ij_star <- t(p_star) + p_star

    # step 3n + 1
    for (i in 1:r) {
      for (j in 1:r) {
        if (truncated && i == j) {
          next
        }
        p_star[i, j] <- t_ij[i, j] * p_star[i, j] / t_ij_star[i, j]
      }
    }

    # step 3n + 2
    p_star_i_dot <- rowSums(p_star)
    for (i in 1:r) {
      for (j in 1:r) {
        if (truncated && i == j) {
          next
        }
        p_star[i, j] <- p_i_dot[i] * p_star[i, j] / p_star_i_dot[i]
      }
    }

    # step 3n + 3
    p_star_dot_j <- colSums(p_star)
    for (i in 1:r) {
      for (j in 1:r) {
        if (truncated && i == j) {
          next
        }
        p_star[i, j] <- p_dot_j[j] * p_star[i, j] / p_star_dot_j[j]
      }
    }

    if (verbose) {
      max_diff <- 0.0
      for (i in 1:r) {
        for (j in 1:r) {
          if (truncated && i == j) {
            next
          }
          max_diff <- max(abs(initial[i, j] - p_star[i, j]), max_diff)
        }
      }
      message(paste(iter, max_diff))
    }
  }

  x_star <- N * p_star
  if (verbose) {
    message(x_star)
  }
  mdis <- Ireland_mdis(n, x_star, truncated)
  df <- r - 1
  list(mdis=mdis, df=df, x_star=x_star, p_star=p_star)
}
