#' @title 4PL Logistic Dose-Response Model (Emax/Imax)
#' @name logistic_4pl
#' @description
#' Fits pharmacodynamic dose-response data to a 4-parameter logistic (4PL) model
#' using nonlinear least squares regression.
#'
#' The model can handle **both increasing (Stimulatory / Emax)** and
#' **decreasing (Inhibitory / Imax)** dose-response curves automatically.
#'
#' The 4PL model generalizes the Hill/Sigmoid model:
#'
#' \deqn{
#'   E = E_{min} + \frac{E_{max} - E_{min}}{1 + (EC_{50} / D)^n} \quad \text{for increasing curves}
#' }
#'
#' \deqn{
#'   E = E_{min} + \frac{E_{max} - E_{min}}{1 + (D / IC_{50})^n} \quad \text{for decreasing curves}
#' }
#'
#' Key features:
#' * Automatically detects increasing vs decreasing responses
#' * \eqn{E_{min}} = minimum response (floor)
#' * \eqn{E_{max}} = maximum response (ceiling)
#' * \eqn{EC_{50}/IC_{50}} = dose producing half-maximal effect
#' * \eqn{n} = Hill coefficient (steepness)
#' * Symmetric sigmoid curve about midpoint
#'
#' @param data A data frame containing dose-response experimental data.
#' @param dose_col Character string specifying the column name for dose.
#' @param response_col Character string specifying the column name for measured response.
#' @param group_col Optional character string specifying a column for grouping.
#' @param log_dose Logical; if TRUE, dose values are log10-transformed for plotting.
#' @param plot Logical; if TRUE, generates a dose-response plot with fitted 4PL curves.
#' @param annotate Logical; if TRUE, annotates the plot with model parameters and fit metrics.
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats na.omit median nls nls.control AIC BIC
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text labs theme
#' theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with E_min, E_max, EC50, n, RMSE, AIC, and BIC for each group.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single increasing curve (Emax)
#' df_emax <- data.frame(
#'   dose = c(0.1, 0.3, 1, 3, 10, 30, 100),
#'   response = c(5, 12, 28, 55, 75, 90, 98)
#' )
#' logistic_4pl(
#'   data = df_emax,
#'   dose_col = "dose",
#'   response_col = "response"
#' )
#'
#' # Example II: Single decreasing curve (Imax)
#' df_imax <- data.frame(
#'   dose = c(0.1, 0.3, 1, 3, 10, 30, 100),
#'   response = c(95, 88, 70, 50, 30, 15, 5)
#' )
#' logistic_4pl(
#'   data = df_imax,
#'   dose_col = "dose",
#'   response_col = "response"
#' )
#'
#' # Example III: Two treatment groups, mixed Emax/Imax
#' df_groups <- data.frame(
#'   dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 2),
#'   response = c(
#'     4, 10, 25, 55, 78, 92,   # Group A: increasing (Emax)
#'     90, 75, 55, 35, 20, 10   # Group B: decreasing (Imax)
#'   ),
#'   treatment = rep(c("Group A", "Group B"), each = 6)
#' )
#' logistic_4pl(
#'   data = df_groups,
#'   dose_col = "dose",
#'   response_col = "response",
#'   group_col = "treatment",
#'   log_dose = TRUE
#' )
#' @references Holford, N. H. G. & Sheiner, L. B. (1981) <doi:10.2165/00003088-198106060-00002>
#' Understanding the dose-effect relationship. Clinical Pharmacokinetics, 6(6), 429–453.
#' @references Finney, D. J. (1971) <isbn:9780521080415> Probit Analysis, 3rd Edition.
#' Cambridge University Press, Cambridge.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("dose", "response", "group", "plot_dose", "E_min", "E_max",
                         "EC50", "n", "RMSE", "AIC", "BIC", "label", "x", "y"))

logistic_4pl <- function(data,
                         dose_col="dose",
                         response_col="response",
                         group_col=NULL,
                         log_dose=FALSE,
                         plot=TRUE,
                         annotate=TRUE) {

  if (!requireNamespace("ggplot2", quietly=TRUE)) stop("Package 'ggplot2' required.")

  df <- data[, c(dose_col, response_col, group_col), drop=FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("dose","response")
  df$group <- if (!is.null(group_col)) as.factor(df[[group_col]]) else factor("Experimental")

  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    increasing <- as.numeric(d$response[which.max(d$dose)]) > as.numeric(d$response[which.min(d$dose)])
    start_vals <- list(
      E_min = min(d$response),
      E_max = max(d$response),
      EC50 = stats::median(d$dose),
      n = 1
    )

    # Swap E_min/E_max for decreasing curves
    if (!increasing) {
      start_vals$E_min <- max(d$response)
      start_vals$E_max <- min(d$response)
    }

    fit <- try(
      stats::nls(response ~ E_min + (E_max - E_min) / (1 + (EC50 / dose)^n),
                 data=d, start=start_vals, control=stats::nls.control(maxiter=200)),
      silent=TRUE
    )

    if (inherits(fit, "try-error")) {
      return(data.frame(group=unique(d$group), E_min=NA, E_max=NA, EC50=NA, n=NA, RMSE=NA, AIC=NA, BIC=NA))
    }

    coef_fit <- coef(fit)
    pred <- predict(fit)
    RMSE <- sqrt(mean((d$response - pred)^2))
    AIC_val <- stats::AIC(fit)
    BIC_val <- stats::BIC(fit)

    data.frame(group=unique(d$group),
               E_min=coef_fit["E_min"],
               E_max=coef_fit["E_max"],
               EC50=coef_fit["EC50"],
               n=coef_fit["n"],
               RMSE=RMSE,
               AIC=AIC_val,
               BIC=BIC_val)
  }))

  # Generate fitted curves
  df_fit <- do.call(rbind, lapply(split(df, df$group), function(d) {
    fr <- fit_results[fit_results$group==unique(d$group),]
    dose_seq <- seq(min(d$dose), max(d$dose), length.out=200)
    fitted <- fr$E_min + (fr$E_max - fr$E_min) / (1 + (fr$EC50 / dose_seq)^fr$n)
    data.frame(dose=dose_seq, response=fitted, group=unique(d$group))
  }))

  if (log_dose) {
    df$plot_dose <- log10(df$dose)
    df_fit$plot_dose <- log10(df_fit$dose)
    xlab <- "Log10 Dose"
  } else {
    df$plot_dose <- df$dose
    df_fit$plot_dose <- df_fit$dose
    xlab <- "Dose"
  }

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x=plot_dose, y=response, color=group)) +
      ggplot2::geom_point(size=3) +
      ggplot2::geom_line(data=df_fit, linewidth=1.2) +
      ggplot2::labs(title="4PL Logistic Dose-Response Curve", x=xlab, y="Response") +
      ggplot2::theme_bw(base_size=14) +
      ggplot2::theme(panel.grid=ggplot2::element_blank(),
                     plot.title=ggplot2::element_text(face="bold", hjust=0.5))

    if (annotate && nlevels(df$group)<=1) {
      ann <- fit_results
      ann$label <- paste0("E_min=", round(ann$E_min,2), "\nE_max=", round(ann$E_max,2),
                          "\nEC50=", round(ann$EC50,2), "\nn=", round(ann$n,2),
                          "\nRMSE=", round(ann$RMSE,2),
                          "\nAIC=", round(ann$AIC,2),
                          "\nBIC=", round(ann$BIC,2))
      x_rng <- range(df$plot_dose)
      y_rng <- range(df$response)
      ann$x <- x_rng[2] - 0.05*diff(x_rng)
      ann$y <- y_rng[2] - 0.05*diff(y_rng)
      p <- p + ggplot2::geom_text(data=ann, ggplot2::aes(x=x, y=y, label=label, color=group),
                                  hjust=1, vjust=1, size=4, show.legend=FALSE)
    }

    print(p)
  }

  return(list(fitted_parameters=fit_results, data=df))
}
