#' @title Sigmoid Emax (Stimulatory Hill Model) for Dose-Response Analysis
#' @name sigmoid_emax
#' @description
#' Fits pharmacodynamic dose-response data to the stimulatory Hill
#' (Sigmoid Emax) model using nonlinear least squares regression.
#'
#' The stimulatory Hill model describes increasing pharmacological
#' effects with increasing dose (or concentration) according to:
#'
#' \deqn{
#'   E = E_0 + \frac{E_{max} \cdot D^n}{EC_{50}^n + D^n}
#' }
#'
#' where \eqn{E_0} is the baseline effect, \eqn{E_{max}} is the maximum
#' stimulatory effect, \eqn{EC_{50}} is the dose producing 50% of
#' \eqn{E_{max}}, and \eqn{n} is the Hill coefficient.
#'
#' This model is appropriate when the observed response increases
#' monotonically with dose.
#'
#' @param data A data frame containing dose-response experimental data.
#' @param dose_col Character string specifying the column name for dose or concentration.
#' @param response_col Character string specifying the column name for measured response.
#' @param group_col Optional character string specifying a column for grouping.
#' @param log_dose Logical; if TRUE, dose values are log10-transformed for plotting
#'   (model fitting uses original dose values).
#' @param plot Logical; if TRUE, generates a dose-response plot with fitted Hill curves.
#' @param annotate Logical; if TRUE, annotates the plot with model parameters and fit metrics
#'   (only if <=1 group).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats na.omit median nls nls.control AIC BIC
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text labs theme
#' theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with E0, Emax, EC50, Hill coefficient (\code{n}),
#'         RMSE, AIC, and BIC values for each group.}
#'   \item{\code{data}}{The processed dataset used for model fitting and plotting.}
#' }
#' @examples
#' # Example I: Single dose-response dataset
#' df <- data.frame(
#'   dose = c(0.1, 0.3, 1, 3, 10, 30, 100),
#'   response = c(5, 10, 25, 55, 80, 92, 98)
#' )
#' sigmoid_emax(
#'   data = df,
#'   dose_col = "dose",
#'   response_col = "response"
#' )
#'
#' # Example II: Two treatment groups
#' df2 <- data.frame(
#'   dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 2),
#'   response = c(
#'     3, 8, 20, 45, 70, 85,     # Group A
#'     2, 6, 15, 35, 60, 78      # Group B
#'   ),
#'   treatment = rep(c("Group A", "Group B"), each = 6)
#' )
#' sigmoid_emax(
#'   data = df2,
#'   dose_col = "dose",
#'   response_col = "response",
#'   group_col = "treatment",
#'   log_dose = TRUE
#' )
#'
#' # Example III: Multiple subjects (population-style dose-response pharmacodynamics)
#' df_subjects <- data.frame(
#'   dose = rep(c(0.1, 0.3, 1, 3, 10, 30), 5),
#'   response = c(
#'     5, 13, 30, 56, 80, 92,   # Subject 1
#'     4, 12, 28, 54, 78, 90,   # Subject 2
#'     6, 15, 33, 59, 83, 95,   # Subject 3
#'     5, 14, 31, 57, 81, 93,   # Subject 4
#'     3, 11, 26, 52, 76, 88    # Subject 5
#'   ),
#'   subject = rep(paste0("S", 1:5), each = 6)
#' )
#' sigmoid_emax(
#'   data = df_subjects,
#'   dose_col = "dose",
#'   response_col = "response",
#'   group_col = "subject",
#'   log_dose = TRUE
#' )

#' @references Hill, A. V. (1910) The possible effects of the aggregation of the
#' molecules of hæmoglobin on its dissociation curves. The Journal of Physiology,
#' 40(4), iv–vii.
#' @references Holford, N. H. G. & Sheiner, L. B. (1981) <doi:10.2165/00003088-198106060-00002>
#' Understanding the dose-effect relationship. Clinical Pharmacokinetics, 6(6), 429–453.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("dose", "response", "group", "E0", "Emax", "EC50", "Hill",
                         "RMSE", "AIC", "BIC", "plot_dose", "label", "x", "y"))

sigmoid_emax <- function(data,
                         dose_col = "dose",
                         response_col = "response",
                         group_col = NULL,
                         log_dose = FALSE,
                         plot = TRUE,
                         annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop("Package 'ggplot2' is required.")
  }

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(dose_col, response_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)

  colnames(df)[1:2] <- c("dose", "response")

  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- factor("Experimental")
  }

  # -------------------------
  # Hill model fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    start_vals <- list(
      E0   = min(d$response),
      Emax = max(d$response) - min(d$response),
      EC50 = stats::median(d$dose),
      n    = 1
    )

    fit <- try(
      stats::nls(
        response ~ E0 + (Emax * dose^n) / (EC50^n + dose^n),
        data = d,
        start = start_vals,
        control = stats::nls.control(maxiter = 200)
      ),
      silent = TRUE
    )

    if (inherits(fit, "try-error")) {
      return(data.frame(
        group = unique(d$group),
        E0 = NA_real_,
        Emax = NA_real_,
        EC50 = NA_real_,
        Hill = NA_real_,
        RMSE = NA_real_,
        AIC = NA_real_,
        BIC = NA_real_
      ))
    }

    coef_fit <- coef(fit)
    pred <- predict(fit)

    RMSE <- sqrt(mean((d$response - pred)^2))
    AIC_val <- stats::AIC(fit)
    BIC_val <- stats::BIC(fit)

    data.frame(
      group = unique(d$group),
      E0 = coef_fit["E0"],
      Emax = coef_fit["Emax"],
      EC50 = coef_fit["EC50"],
      Hill = coef_fit["n"],
      RMSE = RMSE,
      AIC = AIC_val,
      BIC = BIC_val
    )
  }))

  # -------------------------
  # Generate fitted curves
  # -------------------------
  df_fit <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fr <- fit_results[fit_results$group == unique(d$group), ]

    dose_seq <- seq(min(d$dose), max(d$dose), length.out = 200)

    fitted <- fr$E0 +
      (fr$Emax * dose_seq^fr$Hill) /
      (fr$EC50^fr$Hill + dose_seq^fr$Hill)

    data.frame(
      dose = dose_seq,
      response = fitted,
      group = unique(d$group)
    )
  }))

  if (log_dose) {
    df$plot_dose <- log10(df$dose)
    df_fit$plot_dose <- log10(df_fit$dose)
    xlab <- "Log10 Dose"
  } else {
    df$plot_dose <- df$dose
    df_fit$plot_dose <- df_fit$dose
    xlab <- "Dose"
  }

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(df, ggplot2::aes(x = plot_dose, y = response, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(data = df_fit, linewidth = 1.2) +
      ggplot2::labs(
        title = "Dose-Response Curve (Hill Model)",
        subtitle = "Sigmoid Emax Pharmacodynamic Model",
        x = xlab,
        y = "Response",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Conditional annotations
    # -------------------------
    num_groups <- nlevels(df$group)

    if (annotate && num_groups <= 1) {

      ann <- fit_results
      ann$label <- paste0(
        "Emax = ", round(ann$Emax, 2), "\n",
        "EC50 = ", round(ann$EC50, 2), "\n",
        "Hill (n) = ", round(ann$Hill, 2), "\n",
        "RMSE = ", round(ann$RMSE, 2), "\n",
        "AIC = ", round(ann$AIC, 1), "\n",
        "BIC = ", round(ann$BIC, 1)
      )

      x_rng <- range(df$plot_dose, na.rm = TRUE)
      y_rng <- range(df$response, na.rm = TRUE)

      ann$x <- x_rng[2] - 0.05 * diff(x_rng)
      ann$y <- y_rng[2] - 0.05 * diff(y_rng)

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(x = x, y = y, label = label, color = group),
          hjust = 1,
          vjust = 1,
          size = 4,
          show.legend = FALSE
        )

    } else if (annotate && num_groups > 2) {
      message("More than 2 groups detected -- annotations disabled to prevent overlap.")
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
