#' @title
#' Checking for downsets.
#'
#' @description
#' Checks whether the input elements form a downset, in the input poset.
#'
#' @param poset An object of S4 class `POSet`.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .

#' @param elements A vector of character strings (the names of the input) elements).
#'
#' @return
#' A boolean value.
#'
#' @examples
#' elems <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)
#'
#' chk <- IsDownset(pos, c("a", "b", "c"))
#'
#' @name IsDownset
#' @export IsDownset
IsDownset <- function(poset, elements) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    if (!is.character(elements))
      stop("element must be a string vector")
    elements <- unique(elements)
    result <- .Call("_IsDownset", poset@ptr, elements)
    return (result)
  }, error = function(err) {
    # error handler picks up where error was generated
    print(paste("MY_ERROR:  ",err))
  }) # END tryCatch
}
