% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/recor.R
\name{recor}
\alias{recor}
\title{Rearrangement Correlation (\eqn{r^\#})}
\usage{
recor(x, y = NULL)
}
\arguments{
\item{x}{A numeric vector, matrix, or data.frame.}

\item{y}{NULL (default) or a vector, matrix, or data.frame with compatible dimensions to x.}
}
\value{
A numeric value or matrix containing the rearrangement correlation coefficient(s):
\itemize{
\item Single correlation value if x and y are vectors
\item Correlation matrix if x is a matrix/data.frame (symmetric if y is NULL,
rectangular if y is provided)
}
}
\description{
Computes the rearrangement correlation coefficient (\eqn{r^\#}), an adjusted version of Pearson's
correlation coefficient that addresses the underestimation problem for monotonic dependence.
This coefficient captures arbitrary monotone relationships (both linear and nonlinear) while
reverting to Pearson's \eqn{r} in linear scenarios. The rearrangement correlation is derived from
a tighter inequality than the classical Cauchy-Schwarz inequality, providing sharper bounds
and expanded capture range.
}
\details{
The rearrangement correlation coefficient is based on rearrangement inequality theorems that
provide tighter bounds than the Cauchy-Schwarz inequality. Mathematically, for samples \eqn{x} and \eqn{y},
it is defined as:

\deqn{r^\#(x, y) = \frac{s_{x,y}}{\left| s_{x^\uparrow, y^\updownarrow} \right|}}

where:
\itemize{
\item \eqn{s_{x,y}} is the sample covariance between \eqn{x} and \eqn{y}.
\item \eqn{x^\uparrow} is the increasing rearrangement of \eqn{x}.
\item \eqn{y^\updownarrow} is either \eqn{y^\uparrow} (increasing rearrangement) if \eqn{s_{x,y} \geq 0}
or \eqn{y^\downarrow} (decreasing rearrangement) if \eqn{s_{x,y} < 0}
}

Key properties of the rearrangement correlation:
\itemize{
\item \eqn{|r^\#| \leq 1} with equality if and only if \eqn{x} and \eqn{y} are monotone dependent.
\item \eqn{|r^\#| \geq |r|} with equality when \eqn{y} is a permutation of \eqn{ax + b}
(i.e., linear relationship).
\item More accurate than classical coefficients for nonlinear monotone dependence.
\item Reverts to Pearson's \eqn{r} for linear relationships and to Spearman's \eqn{\rho} for rank data.
}

The function automatically handles various input types consistently with \code{stats::cor()}:
\itemize{
\item If x is a vector and y is a vector: returns a single correlation value
\item If x is a matrix/data.frame and y is NULL: returns correlation matrix between columns of x
\item If x and y are both matrices/data.frames: returns correlation matrix between columns of x and y
}
}
\examples{
# Vector example (perfect linear relationship)
x <- c(1, 2, 3, 4, 5)
y <- c(2, 4, 6, 8, 10)
recor(x, y) # Returns 1.0

# Nonlinear monotone relationship
x <- c(1, 2, 3, 4, 5)
y <- c(1, 8, 27, 65, 125) # y = x^3
recor(x, y) # Higher value than Pearson's r
cor(x, y)

# Matrix example
set.seed(123)
mat <- matrix(rnorm(100), ncol = 5)
colnames(mat) <- LETTERS[1:5]
recor(mat) # 5x5 correlation matrix

# Two matrices
mat1 <- matrix(rnorm(50), ncol = 5)
mat2 <- matrix(rnorm(50), ncol = 5)
recor(mat1, mat2) # 5x5 cross-correlation matrix

# data.frame
recor(iris[, 1:4]) # correlation matrix for iris dataset features

}
\references{
Ai, X. (2024). Adjust Pearson's r to Measure Arbitrary Monotone Dependence.
In \emph{Advances in Neural Information Processing Systems} (Vol. 37, pp. 37385-37407).
\url{https://proceedings.neurips.cc/paper_files/paper/2024/file/41c38a83bd97ba28505b4def82676ba5-Paper-Conference.pdf}
}
\seealso{
\code{\link{cor}} for Pearson's correlation coefficient,
\code{\link{cor}} with \code{method = "spearman"} for Spearman's rank correlation,
\code{\link{cor}} with \code{method = "kendall"} for Kendall's rank correlation,
\code{\link{cov}} for covariance calculation
}
