#' Set Screenshotbase API key
#'
#' Stores the API key in the R session via options, used by subsequent requests.
#'
#' You can also set environment variable `SCREENSHOTBASE_API_KEY` for automatic discovery.
#'
#' @param key Character scalar, your Screenshotbase API key.
#' @return Invisibly returns the key.
#' @export
set_api_key <- function(key) {
  stopifnot(is.character(key), length(key) == 1, !is.na(key), nzchar(key))
  options(screenshotbase.apikey = key)
  invisible(key)
}

.sb_get_api_key <- function(explicit) {
  if (!is.null(explicit) && nzchar(explicit)) return(explicit)
  key <- getOption("screenshotbase.apikey", NULL)
  if (!is.null(key) && nzchar(key)) return(key)
  key <- Sys.getenv("SCREENSHOTBASE_API_KEY", unset = NA_character_)
  if (!is.na(key) && nzchar(key)) return(key)
  stop("No API key provided. Use set_api_key() or set SCREENSHOTBASE_API_KEY.", call. = FALSE)
}

.sb_base_url <- function() {
  base <- Sys.getenv("SCREENSHOTBASE_BASE_URL", unset = "https://api.screenshotbase.com")
  sub("/*$", "", base)
}

.sb_build_request <- function(endpoint, query = list(), apikey = NULL) {
  key <- .sb_get_api_key(apikey)
  url <- sprintf("%s/v1/%s", .sb_base_url(), endpoint)
  req <- httr2::request(url)
  req <- httr2::req_headers(req, apikey = key)
  if (length(query)) {
    req <- do.call(httr2::req_url_query, c(list(req = req), query))
  }
  req
}


