% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/st_top_publishers.R
\name{st_top_publishers}
\alias{st_top_publishers}
\title{Get Top Publishers by Revenue or Downloads}
\usage{
st_top_publishers(
  measure = "revenue",
  os,
  category = 0,
  time_range = "month",
  comparison_attribute = "absolute",
  date,
  end_date = NULL,
  country,
  limit = 25,
  offset = 0,
  device_type = "total",
  include_apps = TRUE,
  auth_token = Sys.getenv("SENSORTOWER_AUTH_TOKEN")
)
}
\arguments{
\item{measure}{Character. Metric to rank by: "revenue" or "units" (downloads).
Defaults to "revenue".}

\item{os}{Character. Operating system: "ios", "android", or "unified". Required.}

\item{category}{Integer or character. Category ID to filter publishers.
For iOS use numeric IDs (e.g., 6014 for Games), for Android use strings
(e.g., "game"). Use 0 or "all" for all categories.}

\item{time_range}{Character. Time period: "day", "week", "month", "quarter",
or "year". Defaults to "month".}

\item{comparison_attribute}{Character. Data type to return: "absolute"
(total values), "delta" (growth), or "transformed_delta" (growth rate).
Defaults to "absolute".}

\item{date}{Date or character. Start date in "YYYY-MM-DD" format. Required.
**Important**: Must align with time_range boundaries:
- `month`: Must be first day of month (e.g., 2025-06-01)
- `week`: Must be Monday
- `quarter`: Must be quarter start (Jan 1, Apr 1, Jul 1, Oct 1)
- `year`: Must be January 1st
- `day`: Any date allowed
Function will error if date doesn't align. Defaults to 30 days ago.}

\item{end_date}{Date or character. Optional end date for aggregating multiple
periods. If not provided with `time_range = "month"`, "quarter", or "year",
it will be automatically set to the last day of the period. 
**Important**: If provided, must align with time_range boundaries:
- `month`: Must be last day of month (e.g., 2025-06-30, 2025-07-31)
- `week`: Must be Sunday  
- `quarter`: Must be quarter end (Mar 31, Jun 30, Sep 30, Dec 31)
- `year`: Must be December 31st
- `day`: Any date allowed
Function will error if date doesn't align. Use `time_range = "day"`
for custom date ranges.}

\item{country}{Character. Country or region code (e.g., "US", "GB", "WW" for worldwide). Required.}

\item{limit}{Integer. Number of publishers to return (1-100). Defaults to 25.}

\item{offset}{Integer. Number of publishers to skip for pagination. Defaults to 0.}

\item{device_type}{Character. For iOS: "iphone", "ipad", or "total".
For unified: "total". Ignored for Android. Defaults to "total".}

\item{include_apps}{Logical. Whether to include each publisher's top apps
in the response. Defaults to TRUE.}

\item{auth_token}{Character. Your Sensor Tower API authentication token.
Defaults to the value stored in the `SENSORTOWER_AUTH_TOKEN` environment
variable.}
}
\value{
A [tibble][tibble::tibble] containing top publishers with columns:
  - `publisher_id`: Unique publisher identifier
  - `publisher_name`: Publisher display name
  - `date`: Date of the metrics (as provided by API)
  - `date_start`: Actual start date of the period covered
  - `date_end`: Actual end date of the period covered
  - `units_absolute`: Total downloads for the period
  - `units_delta`: Download growth vs previous period
  - `units_transformed_delta`: Download growth rate
  - `revenue_absolute`: Total revenue in cents for the period
  - `revenue_delta`: Revenue growth vs previous period
  - `revenue_transformed_delta`: Revenue growth rate
  - `revenue_usd`: Total revenue in USD (revenue_absolute / 100)
  - `rank`: Publisher rank based on selected measure
  - `apps`: Nested tibble with top apps (if include_apps = TRUE)
}
\description{
Retrieves top app publishers ranked by revenue or downloads for a specified
category, time period, and country. This function uses the
`/v1/\{os\}/top_and_trending/publishers` endpoint.
}
\section{API Notes}{

  - All revenue values are returned in cents from the API
  - The function adds a `revenue_usd` column for convenience
  - Growth metrics compare to the previous equivalent period
  - Worldwide data may have a 2-3 day lag vs country-specific data
}

\section{Date Handling}{

  When using `time_range = "month"`, "quarter", or "year":
  - Dates MUST align with period boundaries or the function will error
  - Example: For `time_range = "month"`, use `date = "2025-06-01"`, not `"2025-06-27"`
  - This prevents unexpected data ranges from the API
  - For custom date ranges (e.g., June 27 - July 26), use `time_range = "day"` and aggregate
}

\examples{
\dontrun{
# Get top 10 game publishers by revenue for last month
top_publishers <- st_top_publishers(
  measure = "revenue",
  category = 6014,  # Games category for iOS
  limit = 10
)

# Get top publishers by downloads with growth metrics
growth_publishers <- st_top_publishers(
  measure = "units",
  comparison_attribute = "delta",
  time_range = "week",
  limit = 20
)

# This will ERROR - dates don't align with month boundaries:
# publishers_custom <- st_top_publishers(
#   date = "2025-06-27",        # ERROR: Not start of month!
#   end_date = "2025-07-26",    # ERROR: Not end of month!
#   time_range = "month"
# )

# Correct way for full months (end_date auto-calculated):
publishers_month <- st_top_publishers(
  date = "2025-06-01",      # First day of June
  time_range = "month"      # end_date auto-set to 2025-06-30
)

# Or specify multiple months:
publishers_months <- st_top_publishers(
  date = "2025-06-01",      # First day of June
  end_date = "2025-07-31",  # Last day of July
  time_range = "month"
)

# For custom date ranges (e.g., June 27 - July 26), use daily:
daily_publishers <- purrr::map_df(
  seq(as.Date("2025-06-27"), as.Date("2025-07-26"), by = "day"),
  ~ st_top_publishers(date = .x, time_range = "day", limit = 50)
) \%>\%
  group_by(publisher_id, publisher_name) \%>\%
  summarise(total_revenue = sum(revenue_usd))
}

}
