% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_segmentation.R
\name{sits_segment}
\alias{sits_segment}
\title{Segment an image}
\usage{
sits_segment(
  cube,
  seg_fn = sits_snic(),
  roi = NULL,
  impute_fn = impute_linear(),
  start_date = NULL,
  end_date = NULL,
  memsize = 4L,
  multicores = 2L,
  output_dir,
  version = "v1",
  progress = TRUE
)
}
\arguments{
\item{cube}{Regular data cube}

\item{seg_fn}{Function to apply the segmentation}

\item{roi}{Region of interest (see below)}

\item{impute_fn}{Imputation function to remove NA values.}

\item{start_date}{Start date for the segmentation}

\item{end_date}{End date for the segmentation.}

\item{memsize}{Memory available for classification (in GB).}

\item{multicores}{Number of cores to be used for classification.}

\item{output_dir}{Directory for output file.}

\item{version}{Version of the output (for multiple
segmentations).}

\item{progress}{Show progress bar?}
}
\value{
A tibble of class 'segs_cube' representing the
segmentation.
}
\description{
Apply a spatial-temporal segmentation on a data cube based on a user defined
segmentation function. The function applies the segmentation algorithm
"seg_fn" to each tile. The output is a vector data cube, which is a data cube
with an additional vector file in "geopackage" format.
}
\note{
Segmentation requires the following steps:
\enumerate{
 \item Create a regular data cube with \code{\link[sits]{sits_cube}} and
       \code{\link[sits]{sits_regularize}};
 \item Run \code{\link[sits]{sits_segment}} to obtain a vector data cube
       with polygons that define the boundary of the segments;
 \item Classify the time series associated to the segments
       with \code{\link[sits]{sits_classify}}, to get obtain
       a vector probability cube;
 \item Use \code{\link[sits]{sits_label_classification}} to label the
     vector probability cube;
 \item Display the results with \code{\link[sits]{plot}} or
       \code{\link[sits]{sits_view}}.
}
   The "roi" parameter defines a region of interest. It can be
   an sf_object, a shapefile, or a bounding box vector with
   named XY values ("xmin", "xmax", "ymin", "ymax") or
   named lat/long values ("lon_min", "lat_min", "lon_max", "lat_max").

   As of version 1.5.4, two segmentation functions are available. The
   preferred option is \code{\link[sits]{sits_snic}}, which implements
   the Simple Non-Iterative Clustering (SNIC) algorithm to generate
   compact and homogeneous superpixels directly from uniformly distributed
   seeds. SNIC avoids the iterative refinement step used in SLIC and is
   generally faster and more memory-efficient, making it suitable for
   large multispectral or multitemporal data cubes.

   The previous function \code{\link[sits]{sits_slic}}, based on the
   Simple Linear Iterative Clustering (SLIC) algorithm as adapted by
   Nowosad and Stepinski for multispectral and multitemporal imagery,
   remains available but is now deprecated and will be removed in a future
   release. SLIC clusters pixels using spectral similarity and
   spatial–temporal proximity to produce nearly uniform superpixels,
   but its iterative nature makes it less efficient for large-scale
   Earth observation workflows.

   The result of \code{sits_segment} is a data cube tibble with an additional
   vector file in the \code{geopackage} format. The location of the vector
   file is included in the data cube tibble in a new column, called
   \code{vector_info}.
}
\examples{
if (sits_run_examples()) {
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    # create a data cube
    cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        data_dir = data_dir
    )
    # segment the vector cube
    segments <- sits_segment(
        cube = cube,
        seg_fn = sits_snic(
            grid_seeding = "diamond",
            spacing = 15,
            compactness = 0.5,
            padding = 2
        ),
        output_dir = tempdir()
    )
    # create a classification model
    rfor_model <- sits_train(samples_modis_ndvi, sits_rfor())
    # classify the segments
    seg_probs <- sits_classify(
        data = segments,
        ml_model = rfor_model,
        output_dir = tempdir()
    )
    # label the probability segments
    seg_label <- sits_label_classification(
        cube = seg_probs,
        output_dir = tempdir()
    )
}
}
\references{
Achanta, Radhakrishna, and Sabine Susstrunk. 2017.
        “Superpixels and Polygons Using Simple Non-Iterative Clustering.”
        Proceedings of the IEEE Conference on Computer Vision and Pattern
        Recognition, 4651–60.

        Achanta, Radhakrishna, Appu Shaji, Kevin Smith, Aurelien Lucchi,
        Pascal Fua, and Sabine Süsstrunk. 2012. “SLIC Superpixels Compared
        to State-of-the-Art Superpixel Methods.” IEEE Transactions on
        Pattern Analysis and Machine Intelligence 34 (11): 2274–82.

        Nowosad, Jakub, and Tomasz F. Stepinski. 2022. “Extended SLIC
        Superpixels Algorithm for Applications to Non-Imagery Geospatial
        Rasters.” International Journal of Applied Earth Observation
        and Geoinformation 112 (August): 102935.
}
\author{
Gilberto Camara, \email{gilberto.camara@inpe.br}

Rolf Simoes, \email{rolfsimoes@gmail.com}

Felipe Carvalho, \email{felipe.carvalho@inpe.br}

Felipe Carlos, \email{efelipecarlos@gmail.com}
}
