#' This class represents an SQL field.
#'
#' Used to define a field to be used inside a SELECT or UPDATE statement.
#'
#' @examples
#' # To generate the reference to field "title" in table "books":
#' ExprField$new("title", tabl="books")
#'
#' @import R6
#' @import chk
#' @include Expr.R
#' @include TokenIdentifier.R
#' @include TokenSymbol.R
#' @export
ExprField <- R6::R6Class("ExprField",
  inherit = Expr,
  public = list(

    #' @description
    #' Initializer.
    #' @param name The field name.
    #' @param tabl The table name.
    #' @return Nothing.
    initialize = function(name, tabl = NULL) {
      chk::chk_string(name)
      if (name == "") {
        stop("Field name cannot be empty.", call. = FALSE)
      }
      private$name <- name

      if (!is.null(tabl)) {
        chk::chk_string(tabl)
      }
      private$table <- tabl

      return(invisible(NULL))
    },

    #' @description
    #' Return the associted table.
    #' @return The associated table, as a character value, NA if no table is
    #'         defined.
    getTable = function() {
      return(if (is.null(private$table)) NA_character_ else private$table)
    },

    #' @description
    #' Generate the list of tokens representing this statement.
    #' @return A list of Token objects.
    getTokens = function() {
      tokens <- list()
      if (!is.null(private$table)) {
        tokens <- c(tokens, TokenIdentifier$new(private$table), .dot)
      }
      tokens <- c(tokens, TokenIdentifier$new(private$name))
      return(tokens)
    }
  ),
  private = list(
    table = NULL,
    name = NULL
  )
)
