#' @export
#'
#' @title Simulate stochastic correlation model
#' @name stochcorr.sim
#' @description `stochcorr.sim` returns the paths of stock price under a stochastic correlation model
#' @usage stochcorr.sim(m=500, n, dt, S1_0, S2_0, mu1, sigma1, mu2, sigma2,
#' mu, lambda, sigma, corr_process)
#' @details This function returns the simulated paths of two stock prices following a stochastic correlation model. See [stochcorr()]
#' details of the stochastic correlation model
#' @param m number of paths (Default is 500)
#' @param n number of steps in each simulated path
#' @param dt time step
#' @param S1_0 initial price of the first asset
#' @param S2_0 initial price of the second asset
#' @param mu1 drift of the first asset
#' @param sigma1 volatility of the first asset
#' @param mu2 drift of the second asset
#' @param sigma2 volatility of the second asset
#' @param mu mean direction of the correlation process (if `corr_process=vmp`)
#' @param lambda drift of the correlation process (if `corr_process=vmp`)
#' @param sigma volatility of the correlation process (if `corr_process=vmp` or `corr_process=cbm`)
#' @param corr_process specify the correlation process, `vmp` for von Mises process or `cbm` for Circular Brownian Motion
#' @return Returns a list with prices of two assets `S1` and `S2` under the stochastic correlation model
#' @examples
#' library(stochcorr)
#' # Generate 500 paths of two geometric Brownian motions, S1 and S2, of length 100 each
#' # following the von Mises process with mu=pi/2, lambda=1 and sigma =1
#'
#' a<-stochcorr.sim(m=500,100,0.01,100,100,0.05,0.05,0.06,0.1,pi/2,1,1,"vmp")
#' t<-seq(0,100*0.01-0.01,0.01)
#'
#' # Plot the first realization of S1 and S2
#'
#' plot(t,a$S1[1,], ylim=c(min(a$S1[1,],a$S2[1,]),max(a$S1[1,],a$S2[1,])),type="l")
#' lines(t,a$S2[1,], col="red",type="l")
#' legend(0.01,max(a$S1[1,],a$S2[1,]), legend = c("S1","S2"), col = c("black", "red"), lty=1)
#'

stochcorr.sim<-function(m=500,n,dt,S1_0,S2_0,mu1,sigma1,mu2,sigma2,mu,lambda,sigma,corr_process){
  if(corr_process=="vmp"){
    corr<-cos(rtraj.vmp(n,mu,dt,mu,lambda,sigma))
    S<-sim_stochcorr(dt,S1_0,S2_0,mu1,mu2,sigma1,sigma2,corr,iter=m)
    return(S)
  } else if(corr_process=="cbm"){
    corr<-cos(rtraj.cbm(n,0,dt,sigma))
    S<-sim_stochcorr(dt,S1_0,S2_0,mu1,mu2,sigma1,sigma2,corr,iter=m)
    return(S)
  } else {
    stop("incorrect corr_process")
  }
}
