#[cfg(not(feature = "std"))]
use alloc::vec::Vec;
use core::ptr::NonNull;
use std::mem;
use std::mem::MaybeUninit;

#[allow(unused_imports)] // Needed for Rust 1.64
use rawpointer::PointerExt;

use crate::imp_prelude::*;

use crate::dimension;
use crate::error::{ErrorKind, ShapeError};
use crate::iterators::Baseiter;
use crate::low_level_util::AbortIfPanic;
use crate::OwnedRepr;
use crate::Zip;




impl<A> Array<A, Ix0>
{
    
    
    
    
    
    
    /// #[derive(Debug, Eq, PartialEq)]
    
    
    
    
    
    
    pub fn into_scalar(self) -> A
    {
        let size = mem::size_of::<A>();
        if size == 0 {
            
            self.data.into_vec().remove(0)
        } else {
            
            
            
            
            let first = self.ptr.as_ptr() as usize;
            let base = self.data.as_ptr() as usize;
            let index = (first - base) / size;
            debug_assert_eq!((first - base) % size, 0);
            
            self.data.into_vec().remove(index)
        }
    }
}




impl<A, D> Array<A, D>
where D: Dimension
{
    
    
    fn offset_from_alloc_to_logical_ptr(&self) -> Option<usize>
    {
        if self.is_empty() {
            return None;
        }
        if std::mem::size_of::<A>() == 0 {
            Some(dimension::offset_from_low_addr_ptr_to_logical_ptr(&self.dim, &self.strides))
        } else {
            let offset = unsafe { self.as_ptr().offset_from(self.data.as_ptr()) };
            debug_assert!(offset >= 0);
            Some(offset as usize)
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn into_raw_vec_and_offset(self) -> (Vec<A>, Option<usize>)
    {
        let offset = self.offset_from_alloc_to_logical_ptr();
        (self.data.into_vec(), offset)
    }

    
    
    
    
    
    
    #[deprecated(note = "Use .into_raw_vec_and_offset() instead", since = "0.16.0")]
    pub fn into_raw_vec(self) -> Vec<A>
    {
        self.into_raw_vec_and_offset().0
    }
}




impl<A> Array<A, Ix2>
{
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn push_row(&mut self, row: ArrayView<A, Ix1>) -> Result<(), ShapeError>
    where A: Clone
    {
        self.append(Axis(0), row.insert_axis(Axis(0)))
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn push_column(&mut self, column: ArrayView<A, Ix1>) -> Result<(), ShapeError>
    where A: Clone
    {
        self.append(Axis(1), column.insert_axis(Axis(1)))
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn reserve_rows(&mut self, additional: usize) -> Result<(), ShapeError>
    {
        self.reserve(Axis(0), additional)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn reserve_columns(&mut self, additional: usize) -> Result<(), ShapeError>
    {
        self.reserve(Axis(1), additional)
    }
}

impl<A, D> Array<A, D>
where D: Dimension
{
    
    
    
    
    
    
    
    
    /// ## Example
    
    
    
    
    
    
    
    
    
    pub fn move_into<'a, AM>(self, new_array: AM)
    where
        AM: Into<ArrayViewMut<'a, A, D>>,
        A: 'a,
    {
        
        let new_array = new_array.into();
        if mem::needs_drop::<A>() {
            self.move_into_needs_drop(new_array);
        } else {
            
            
            unsafe { self.move_into_uninit(new_array.into_maybe_uninit()) }
        }
    }

    fn move_into_needs_drop(mut self, new_array: ArrayViewMut<A, D>)
    {
        
        
        
        
        Zip::from(&mut self)
            .and(new_array)
            .for_each(|src, dst| mem::swap(src, dst));
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    /// ## Example
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn move_into_uninit<'a, AM>(self, new_array: AM)
    where
        AM: Into<ArrayViewMut<'a, MaybeUninit<A>, D>>,
        A: 'a,
    {
        
        self.move_into_impl(new_array.into())
    }

    fn move_into_impl(mut self, new_array: ArrayViewMut<MaybeUninit<A>, D>)
    {
        unsafe {
            
            let guard = AbortIfPanic(&"move_into: moving out of owned value");
            
            
            Zip::from(self.raw_view_mut())
                .and(new_array)
                .for_each(|src, dst| {
                    src.copy_to_nonoverlapping(dst.as_mut_ptr(), 1);
                });
            guard.defuse();
            
            self.drop_unreachable_elements();
        }
    }

    
    
    
    
    
    /// # Safety
    
    
    fn drop_unreachable_elements(mut self) -> OwnedRepr<A>
    {
        let self_len = self.len();

        
        
        let data_len = self.data.len();

        let has_unreachable_elements = self_len != data_len;
        if !has_unreachable_elements || mem::size_of::<A>() == 0 || !mem::needs_drop::<A>() {
            unsafe {
                self.data.set_len(0);
            }
            self.data
        } else {
            self.drop_unreachable_elements_slow()
        }
    }

    #[inline(never)]
    #[cold]
    fn drop_unreachable_elements_slow(mut self) -> OwnedRepr<A>
    {
        
        
        let data_len = self.data.len();
        let data_ptr = self.data.as_nonnull_mut();

        unsafe {
            
            
            let self_ = self.raw_view_mut();
            self.data.set_len(0);

            drop_unreachable_raw(self_, data_ptr, data_len);
        }

        self.data
    }

    
    
    
    pub(crate) fn empty() -> Array<A, D>
    {
        assert_ne!(D::NDIM, Some(0));
        let ndim = D::NDIM.unwrap_or(1);
        Array::from_shape_simple_fn(D::zeros(ndim), || unreachable!())
    }

    
    #[cold]
    fn change_to_contig_append_layout(&mut self, growing_axis: Axis)
    {
        let ndim = self.ndim();
        let mut dim = self.raw_dim();

        
        
        
        let mut new_array;
        if growing_axis == Axis(ndim - 1) {
            new_array = Self::uninit(dim.f());
        } else {
            dim.slice_mut()[..=growing_axis.index()].rotate_right(1);
            new_array = Self::uninit(dim);
            new_array.dim.slice_mut()[..=growing_axis.index()].rotate_left(1);
            new_array.strides.slice_mut()[..=growing_axis.index()].rotate_left(1);
        }

        
        
        
        let old_self = std::mem::replace(self, Self::empty());
        old_self.move_into_uninit(new_array.view_mut());

        
        unsafe {
            *self = new_array.assume_init();
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn push(&mut self, axis: Axis, array: ArrayView<A, D::Smaller>) -> Result<(), ShapeError>
    where
        A: Clone,
        D: RemoveAxis,
    {
        
        self.append(axis, array.insert_axis(axis).into_dimensionality::<D>().unwrap())
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn append(&mut self, axis: Axis, mut array: ArrayView<A, D>) -> Result<(), ShapeError>
    where
        A: Clone,
        D: RemoveAxis,
    {
        if self.ndim() == 0 {
            return Err(ShapeError::from_kind(ErrorKind::IncompatibleShape));
        }

        let current_axis_len = self.len_of(axis);
        let self_dim = self.raw_dim();
        let array_dim = array.raw_dim();
        let remaining_shape = self_dim.remove_axis(axis);
        let array_rem_shape = array_dim.remove_axis(axis);

        if remaining_shape != array_rem_shape {
            return Err(ShapeError::from_kind(ErrorKind::IncompatibleShape));
        }

        let len_to_append = array.len();

        let mut res_dim = self_dim;
        res_dim[axis.index()] += array_dim[axis.index()];
        let new_len = dimension::size_of_shape_checked(&res_dim)?;

        if len_to_append == 0 {
            
            
            
            debug_assert_eq!(self.len(), new_len);
            self.dim = res_dim;
            return Ok(());
        }

        let self_is_empty = self.is_empty();
        let mut incompatible_layout = false;

        
        if !self_is_empty && current_axis_len > 1 {
            
            let axis_stride = self.stride_of(axis);
            if axis_stride < 0 {
                incompatible_layout = true;
            } else {
                for ax in self.axes() {
                    if ax.axis == axis {
                        continue;
                    }
                    if ax.len > 1 && ax.stride.abs() > axis_stride {
                        incompatible_layout = true;
                        break;
                    }
                }
            }
        }

        
        if self.len() != self.data.len() {
            incompatible_layout = true;
        }

        if incompatible_layout {
            self.change_to_contig_append_layout(axis);
            
            debug_assert_eq!(self_is_empty, self.is_empty());
            debug_assert_eq!(current_axis_len, self.len_of(axis));
        }

        let strides = if self_is_empty {
            
            
            
            if axis == Axis(self.ndim() - 1) {
                
                
                res_dim.fortran_strides()
            } else {
                
                
                
                res_dim.slice_mut()[..=axis.index()].rotate_right(1);
                let mut strides = res_dim.default_strides();
                res_dim.slice_mut()[..=axis.index()].rotate_left(1);
                strides.slice_mut()[..=axis.index()].rotate_left(1);
                strides
            }
        } else if current_axis_len == 1 {
            
            let new_stride = self.axes().fold(1, |acc, ax| {
                if ax.axis == axis || ax.len <= 1 {
                    acc
                } else {
                    let this_ax = ax.len as isize * ax.stride.abs();
                    if this_ax > acc {
                        this_ax
                    } else {
                        acc
                    }
                }
            });
            let mut strides = self.strides.clone();
            strides[axis.index()] = new_stride as usize;
            strides
        } else {
            self.strides.clone()
        };

        
        self.reserve(axis, array_dim[axis.index()])?;

        unsafe {
            
            
            
            
            
            
            
            
            
            
            
            
            

            
            let mut tail_strides = strides.clone();
            if tail_strides.ndim() > 1 {
                for i in 0..tail_strides.ndim() {
                    let s = tail_strides[i] as isize;
                    if s < 0 {
                        tail_strides.set_axis(Axis(i), -s as usize);
                        array.invert_axis(Axis(i));
                    }
                }
            }

            
            let tail_ptr = self.data.as_end_nonnull();
            let mut tail_view = RawArrayViewMut::new(tail_ptr, array_dim, tail_strides);

            if tail_view.ndim() > 1 {
                sort_axes_in_default_order_tandem(&mut tail_view, &mut array);
                debug_assert!(tail_view.is_standard_layout(),
                              "not std layout dim: {:?}, strides: {:?}",
                              tail_view.shape(), tail_view.strides());
            }

            
            
            
            
            struct SetLenOnDrop<'a, A: 'a>
            {
                len: usize,
                data: &'a mut OwnedRepr<A>,
            }

            impl<A> Drop for SetLenOnDrop<'_, A>
            {
                fn drop(&mut self)
                {
                    unsafe {
                        self.data.set_len(self.len);
                    }
                }
            }

            let mut data_length_guard = SetLenOnDrop {
                len: self.data.len(),
                data: &mut self.data,
            };

            
            Zip::from(tail_view)
                .and_unchecked(array)
                .debug_assert_c_order()
                .for_each(|to, from| {
                    to.write(from.clone());
                    data_length_guard.len += 1;
                });
            drop(data_length_guard);

            
            self.strides = strides;
            self.dim = res_dim;
        }
        
        debug_assert_eq!(self.data.len(), self.len());
        debug_assert_eq!(self.len(), new_len);
        debug_assert!(self.pointer_is_inbounds());

        Ok(())
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn reserve(&mut self, axis: Axis, additional: usize) -> Result<(), ShapeError>
    where D: RemoveAxis
    {
        debug_assert!(axis.index() < self.ndim());
        let self_dim = self.raw_dim();
        let remaining_shape = self_dim.remove_axis(axis);

        
        let len_to_append = remaining_shape
            .size()
            .checked_mul(additional)
            .ok_or(ShapeError::from_kind(ErrorKind::Overflow))?;

        
        let mut res_dim = self_dim;
        res_dim[axis.index()] += additional;
        let new_len = dimension::size_of_shape_checked(&res_dim)?;

        
        debug_assert_eq!(self.len().checked_add(len_to_append).unwrap(), new_len);

        unsafe {
            
            let data_to_array_offset = if std::mem::size_of::<A>() != 0 {
                self.as_ptr().offset_from(self.data.as_ptr())
            } else {
                0
            };
            debug_assert!(data_to_array_offset >= 0);
            self.ptr = self
                .data
                .reserve(len_to_append)
                .offset(data_to_array_offset);
        }

        debug_assert!(self.pointer_is_inbounds());

        Ok(())
    }
}



/// # Safety



pub(crate) unsafe fn drop_unreachable_raw<A, D>(
    mut self_: RawArrayViewMut<A, D>, data_ptr: NonNull<A>, data_len: usize,
) where D: Dimension
{
    let self_len = self_.len();

    for i in 0..self_.ndim() {
        if self_.stride_of(Axis(i)) < 0 {
            self_.invert_axis(Axis(i));
        }
    }
    sort_axes_in_default_order(&mut self_);
    
    let array_memory_head_ptr = self_.ptr;
    let data_end_ptr = data_ptr.add(data_len);
    debug_assert!(data_ptr <= array_memory_head_ptr);
    debug_assert!(array_memory_head_ptr <= data_end_ptr);

    
    
    
    
    
    
    
    

    
    
    let inner_lane_len;
    if self_.ndim() > 1 && self_.strides.last_elem() == 1 {
        self_.dim.slice_mut().rotate_right(1);
        self_.strides.slice_mut().rotate_right(1);
        inner_lane_len = self_.dim[0];
        self_.dim[0] = 1;
        self_.strides[0] = 1;
    } else {
        inner_lane_len = 1;
    }

    
    
    let mut iter = Baseiter::new(self_.ptr, self_.dim, self_.strides);
    let mut dropped_elements = 0;

    let mut last_ptr = data_ptr;

    while let Some(elem_ptr) = iter.next() {
        
        
        while last_ptr != elem_ptr {
            debug_assert!(last_ptr < data_end_ptr);
            std::ptr::drop_in_place(last_ptr.as_mut());
            last_ptr = last_ptr.add(1);
            dropped_elements += 1;
        }
        
        last_ptr = elem_ptr.add(inner_lane_len);
    }

    while last_ptr < data_end_ptr {
        std::ptr::drop_in_place(last_ptr.as_mut());
        last_ptr = last_ptr.add(1);
        dropped_elements += 1;
    }

    assert_eq!(data_len, dropped_elements + self_len,
               "Internal error: inconsistency in move_into");
}




fn sort_axes_in_default_order<S, D>(a: &mut ArrayBase<S, D>)
where
    S: RawData,
    D: Dimension,
{
    if a.ndim() <= 1 {
        return;
    }
    sort_axes1_impl(&mut a.dim, &mut a.strides);
}

fn sort_axes1_impl<D>(adim: &mut D, astrides: &mut D)
where D: Dimension
{
    debug_assert!(adim.ndim() > 1);
    debug_assert_eq!(adim.ndim(), astrides.ndim());
    
    let mut changed = true;
    while changed {
        changed = false;
        for i in 0..adim.ndim() - 1 {
            let axis_i = i;
            let next_axis = i + 1;

            
            debug_assert!(astrides.slice()[axis_i] as isize >= 0);
            if (astrides.slice()[axis_i] as isize) < astrides.slice()[next_axis] as isize {
                changed = true;
                adim.slice_mut().swap(axis_i, next_axis);
                astrides.slice_mut().swap(axis_i, next_axis);
            }
        }
    }
}





fn sort_axes_in_default_order_tandem<S, S2, D>(a: &mut ArrayBase<S, D>, b: &mut ArrayBase<S2, D>)
where
    S: RawData,
    S2: RawData,
    D: Dimension,
{
    if a.ndim() <= 1 {
        return;
    }
    sort_axes2_impl(&mut a.dim, &mut a.strides, &mut b.dim, &mut b.strides);
}

fn sort_axes2_impl<D>(adim: &mut D, astrides: &mut D, bdim: &mut D, bstrides: &mut D)
where D: Dimension
{
    debug_assert!(adim.ndim() > 1);
    debug_assert_eq!(adim.ndim(), bdim.ndim());
    
    let mut changed = true;
    while changed {
        changed = false;
        for i in 0..adim.ndim() - 1 {
            let axis_i = i;
            let next_axis = i + 1;

            
            debug_assert!(astrides.slice()[axis_i] as isize >= 0);
            if (astrides.slice()[axis_i] as isize) < astrides.slice()[next_axis] as isize {
                changed = true;
                adim.slice_mut().swap(axis_i, next_axis);
                astrides.slice_mut().swap(axis_i, next_axis);
                bdim.slice_mut().swap(axis_i, next_axis);
                bstrides.slice_mut().swap(axis_i, next_axis);
            }
        }
    }
}
