/**
 * Some system and generic functions

 * Copyright (C) 2003  Shawn Betts
 * Copyright (C) 2003, 2004  Sylvain Beucler
 * Copyright (C) 1997, 1998, 1999, 2002, 2003  Seth A. Robinson

 * This file is part of GNU FreeDink

 * GNU FreeDink is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.

 * GNU FreeDink is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with program; see the file COPYING. If not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.

 * $Header: /cvsroot/dink/dink/src/util.c,v 1.3 2004/01/16 19:55:45 Beuc Exp $
 */

#include <ctype.h>

/* For _getcwd, _chdir */
#ifdef WIN32
#include <direct.h>
#endif

#include "dink.h"

struct mouse_status_t mouse_status;

void
init_mouse_status (struct mouse_status_t *ms)
{
  ms->x = 0;
  ms->y = 0;
  ms->button = 0;
}

void
SetRect (struct rect * rec, int l, int t, int r, int b)
{
  rec->left = l;
  rec->top = t;
  rec->right = r;
  rec->bottom = b;
}

void
OffsetRect (struct rect * rec, int ofsx, int ofsy)
{
  rec->left += ofsx;
  rec->right += ofsx;
  rec->top += ofsy;
  rec->bottom += ofsy;
}

int
Bottom (SDL_Rect * r)
{
  return r->x + r->h;
}

int
Right (SDL_Rect * r)
{
  return r->x + r->h;
}

int
Top (SDL_Rect * r)
{
  return r->y;
}

int
Left (SDL_Rect * r)
{
  return r->x;
}


void
Offset_Left (SDL_Rect * r, int a)
{
  r->x += a;
  r->w -= a;
}

void
Offset_Top (SDL_Rect * r, int a)
{
  r->y += a;
  r->h -= a;
}

void
Offset_Bottom (SDL_Rect * r, int a)
{
  r->h += a;
}

void
Offset_Right (SDL_Rect * r, int a)
{
  r->w += a;
}

void
CopyRect2 (SDL_Rect * dest, struct rect * src)
{
  dest->x = src->left;
  dest->y = src->top;
  dest->w = src->right - src->left;
  dest->h = src->bottom - src->top;
}

void
CopyRect3 (struct rect * dest, SDL_Rect * src)
{
  dest->left = src->x;
  dest->top = src->y;
  dest->right = src->x + src->w;
  dest->bottom = src->y + src->h;
}


void
CopyRect (struct rect * dest, struct rect * src)
{
  dest->left = src->left;
  dest->right = src->right;
  dest->top = src->top;
  dest->bottom = src->bottom;
}

void
InflateRect (struct rect * r, int x, int y)
{
  r->left -= x;
  r->top -= y;
  r->right += x;
  r->bottom += y;
}

void
BltFast (SDL_Surface * dest, int x, int y, SDL_Surface * src, struct rect * bound)
{
  SDL_Rect bound2;
  SDL_Rect loc;

  loc.x = x;
  loc.y = y;

  if (bound)
    {
      CopyRect2 (&bound2, bound);
      SDL_BlitSurface (src, &bound2, dest, &loc);
    }
  else
    {
      SDL_BlitSurface (src, NULL, dest, &loc);
    }
}

void
Blt (SDL_Surface * dest, struct rect * dest_rect, SDL_Surface * src, struct rect * src_rect)
{
  SDL_Rect *dest_r = NULL, *src_r = NULL;

  if (dest_rect)
    {
      dest_r = (SDL_Rect *) malloc (sizeof (SDL_Rect));
      CopyRect2 (dest_r, dest_rect);
    }

  if (src_rect)
    {
      src_r = (SDL_Rect *) malloc (sizeof (SDL_Rect));
      CopyRect2 (src_r, src_rect);
    }

  SDL_BlitSurface (src, src_r, dest, dest_r);

  if (dest_r)
    free (dest_r);
  if (src_r)
    free (src_r);
}


SDL_Surface *
load_image (char *file, /*bool*/int key, Uint8 r, Uint8 g, Uint8 b)
{
  SDL_Surface *image, *tmp;

/* tmp = SDL_LoadBMP (file); */
  tmp = IMG_Load (file);
  if (tmp == NULL)
    {
      Msg (("Cannot load %s:%s", file, SDL_GetError ()));
      return NULL;
    }

  if (key)
    SDL_SetColorKey (tmp, SDL_SRCCOLORKEY, map_rgb (tmp, r, g, b));

  image = SDL_DisplayFormat (tmp);
  SDL_FreeSurface (tmp);

  if (image == NULL)
    {
      Msg (("Cannot convert surface"));
      return NULL;
    }

  return image;
}

char *
ltoa (int l, char *buf, int base)
{
  sprintf (buf, "%d", l);
  return buf;
}

Uint32
map_rgb (SDL_Surface * surface, Uint8 r, Uint8 g, Uint8 b)
{
  return SDL_MapRGB (surface->format, r, g, b);
}

void
print_text (TTF_Font * font, char *str, int x, int y, SDL_Color /*&*/color,
	    /*bool*/int hcenter)
{
  int new_x, w, h;
  SDL_Surface *tmp, *image;

  if (strlen (str) == 0)
    return;

/* Msg (("printing \"%s\"", str)); */

  tmp = TTF_RenderText_Solid (font, str, color);
  if (tmp == NULL)
    {
      Msg (("Error rendering text"));
    }

  TTF_SizeText (font, str, &w, &h);
  new_x = x;
  if (hcenter)
    new_x -= w / 2;

  image = SDL_DisplayFormat (tmp);
  BltFast (lpDDSBack, new_x, y, tmp, NULL);

  SDL_FreeSurface (image);
  SDL_FreeSurface (tmp);
}

static SDL_Color text_color;

void
SetTextColor (Uint8 r, Uint8 g, Uint8 b)
{
  text_color.r = r;
  text_color.g = g;
  text_color.b = b;
}

SDL_Color GetTextColor ()
{
  return text_color;
}

void
fill_rect (SDL_Surface * surface, struct rect * box, int color)
{
  SDL_Rect b;

  if (box)
    {
      CopyRect2 (&b, box);
      SDL_FillRect (surface, &b,
		    SDL_MapRGB (surface->format, real_pal[color].r,
				real_pal[color].g, real_pal[color].b));
    }
  else
    {
      SDL_FillRect (surface, NULL,
		    SDL_MapRGB (surface->format, real_pal[color].r,
				real_pal[color].g, real_pal[color].b));
    }
}


void
load_palette_from_bmp (char *file, SDL_Color * palette)
{
  int i;
  SDL_Surface *bmp;

/* bmp = SDL_LoadBMP (file); */
  bmp = IMG_Load (file);

  for (i = 0; i < bmp->format->palette->ncolors; i++)
    {
      palette[i].r = bmp->format->palette->colors[i].r;
      palette[i].g = bmp->format->palette->colors[i].g;
      palette[i].b = bmp->format->palette->colors[i].b;
    }

  SDL_FreeSurface (bmp);
}

void
dink_getcwd (char *buf, int len)
{
#ifdef WIN32
  _getcwd (buf, len);
#else
  getcwd (buf, len);
#endif
}

int
dink_chdir (char *buf)
{
#ifdef WIN32
  return _chdir (buf);
#else
  return chdir (buf);
#endif
}

void
setup_palette ()
{
  int ncolors, i;
  int r, g, b;

  /* Allocate 256 color palette */
  ncolors = 256;

  /* Set a 3,3,2 color cube */
  for (r = 0; r < 8; ++r)
    {
      for (g = 0; g < 8; ++g)
	{
	  for (b = 0; b < 4; ++b)
	    {
	      i = ((r << 5) | (g << 2) | b);
	      real_pal[i].r = r << 5;
	      real_pal[i].g = g << 5;
	      real_pal[i].b = b << 6;
	    }
	}
    }
  /* Note: A better way of allocating the palette might be
     to calculate the frequency of colors in the image
     and create a palette based on that information.
   */
}

int
init_video ()
{
  char *window_title;

  Uint32 flags = SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_TIMER | SDL_INIT_JOYSTICK;
  if (SDL_Init(flags) < 0)
    {
      printf ("Could not initialize SDL: %s.\n", SDL_GetError ());
      exit (1);
    }
  atexit (SDL_Quit);

  windowed = 0;

  window_title = (char *) malloc(strlen(PACKAGE) + 2 + strlen(VERSION) + 1);
  sprintf(window_title, "%s v%s", PACKAGE, VERSION);
  SDL_WM_SetCaption(window_title, NULL);
/*   SDL_Surface *icon = IMG_Load("../w/dink.ico"); */
/*   if (icon == NULL) { */
/*     fprintf (stderr, "Ae: %s\n", */
/* 	     SDL_GetError ()); */
/*     exit(1); */
/*   } */
/*   SDL_WM_SetIcon(icon, NULL); */
  lpDDSPrimary =
    SDL_SetVideoMode (640, 480, 8,
		      SDL_HWPALETTE | SDL_HWSURFACE | SDL_DOUBLEBUF);
  if (lpDDSPrimary == NULL)
    {
      fprintf (stderr, "Couldn't set 640x480x8 video mode: %s\n",
	       SDL_GetError ());
      exit (1);
    }

  lpDDSBack = lpDDSPrimary;

  /* Hide the mouse */
  SDL_ShowCursor (SDL_DISABLE);

  /* Load the palette */
  if (exist ("tiles/splash.bmp"))
    {
      load_palette_from_bmp ("tiles/splash.bmp", real_pal);
    }
  else if (exist ("../dink/tiles/splash.bmp"))
    {
      load_palette_from_bmp ("../dink/tiles/splash.bmp", real_pal);
    }
  else
    {
      Msg (("Failed to create palette from file. Using stock palette"));
      setup_palette ();
    }

  /* Make sure entry 0 is black and 255 is white */
  real_pal[0].r = 0;
  real_pal[0].g = 0;
  real_pal[0].b = 0;
  real_pal[255].r = 255;
  real_pal[255].g = 255;
  real_pal[255].b = 255;

  /* Finally, set the palette */
  SDL_SetColors (lpDDSPrimary, real_pal, 0, 256);

  /* Initialize the font system */
  if (TTF_Init () == -1)
    {
      Msg (("TTF_Init: %s", TTF_GetError ()));
      return 0;
    }

  return 1;
}

int
font_len (TTF_Font * font, char *str, int len)
{
  int text_w, text_h;
  char *tmp;

  tmp = (char *) malloc ((len + 1) * sizeof (char));
  strncpy (tmp, str, len);
  tmp[len] = 0;

  TTF_SizeText (font, tmp, &text_w, &text_h);
  free (tmp);

  return text_w;
}

int
process_text_for_wrapping (TTF_Font * font, char *str, struct rect * box)
{
  int i, start, line;

  start = 0;
  i = 0;
  line = 0;
  while (str[i])
    {
      int len;

      /* Skip forward to the end of the word */
      do
	{
	  i++;
	}
      while (str[i] && str[i] != ' ');

      /* If the length of the text from start to i is bigger than the */
      /* box, then draw the text between start and i. */
      len = font_len (font, &str[start], i - start + 1);
      if (len > box->right - box->left || str[i] == 0)
	{
	  if (str[i])
	    str[i] = '\n';
	  /* continue on a new line */
	  line++;
	  start = i + 1;
	}

    }

  return line;
}

int
print_text_wrap (TTF_Font * font, char *str, struct rect * box, SDL_Color color,
		 /*bool*/int hcenter, /*bool*/int vcenter)
{
  int x, y, lines, line;
  char *tmp, *token;
  tmp = malloc(strlen(str) + 1);
  strcpy(tmp, str);
  /*   tmp = strdup (str); */
  lines = process_text_for_wrapping (font, tmp, box);

  token = strtok (tmp, "\n");

  x = box->left;
  if (hcenter)
    x += (box->right - box->left) / 2;
  y = box->top;
  if (vcenter)
    y += ((box->bottom - box->top) - lines * TTF_FontLineSkip (font)) / 2;
  line = 0;
  while (token)
    {
      print_text (font, token, x, y + line, color, hcenter);
      line += TTF_FontLineSkip (font);
      token = strtok (NULL, "\n");
    }

  return line;
}

void
process_string (char *s)
{
  char *ch;

  while ((ch = strchr (s, '\\')))
    *ch = '/';

  strdown (s);
}

