/* -*- Mode: c++ -*- */
/*
 * Copyright 2001,2002,2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GRFFTAVGSINK_H_
#define _GRFFTAVGSINK_H_

#include <VrSink.h>
#include <gr_fft.h>
#include "VrGUI.h"
#include <algorithm>
#include <stdexcept>

extern "C" {
#include <dlfcn.h>
#include <float.h>
#include <math.h>
	   }

#define	PRINT_PEAK	0


#define FFT_XAXIS_NAME			"Frequency (Hz)"
#define FFT_YAXIS_NAME			"Avg Mag Squared (dB)"


template<class iType> 
class GrFFTAvgSink : public VrSink<iType> {

 public:
  GrFFTAvgSink (VrGUILayout *layout,
		double ymin, double ymax,
		int nPoints = DEFAULT_nPoints,
		const char *label = FFT_YAXIS_NAME);
  
  ~GrFFTAvgSink();

  virtual const char *name() { return "GrFFTAvgSink"; }

  virtual void initialize();

  virtual int work3 (VrSampleRange output, 
		     VrSampleRange inputs[], void *i[]);


  static const int DEFAULT_nPoints = 512;	// number of points to plot
  static const int DIVISIONS = 10;		// number of x-axis divisions
  static const int DEFAULT_display_freq = 20;	// Hz
  static const int DEFAULT_ffts_per_display = 5;  // # of ffts per display update
  static const double ALPHA = 1.0 / (4 * DEFAULT_ffts_per_display);

 private:
  gr_fft_complex *d_fft;
  double         *d_xValues;
  double   	 *d_dbValues;		// FFT magnitude in dB
  double	 *d_avgdbValues;	// average FFT magnitude in dB
  float	         *d_window;
  int       	  d_nPoints;		// max number of points to plot
  float	    	  d_axis_offset;
  VrGUIPlot      *d_display;
  double	  d_ymin, d_ymax;	// possible range of sample values
  int    	  d_nextPoint;		// index of next point to generate for graph
  int 		  d_one_or_two;
  VrGUILayout    *d_layout;
  int		  d_increment;
  int		  d_skip_count;
  const char     *d_label;
  int		  d_display_freq;	// how many display updates / sec
  int		  d_ffts_per_display;	// how may FFTs computed / display
  int		  d_fft_count;

  void collectData (iType *i, long count);

  void set_skip_count (){
    d_skip_count = std::max (0, d_increment - d_nPoints);
  }
};

/*****************************************************************************
 * Implementation of template (C++ requires it to be in .h file).
 ****************************************************************************/


/*
 * Creates a new GrFFTAvgSink.
 */
template<class iType>
GrFFTAvgSink<iType>::GrFFTAvgSink (VrGUILayout *layout,
				   double ymin, double ymax, int nPoints,
				   const char *label) : d_label (label)
{
  d_layout = layout;
  d_nPoints = nPoints;

  d_fft = new gr_fft_complex (d_nPoints);
  d_dbValues = new double[d_nPoints];
  d_avgdbValues = new double[d_nPoints];
  memset (d_avgdbValues, 0, d_nPoints * sizeof (double));
  d_xValues = new double[d_nPoints];
  d_window = new float[d_nPoints];

  for (int i = 0; i < d_nPoints; i++)
    d_window[i] = 0.54 - 0.46 * cos (2*M_PI/d_nPoints*i); // hamming d_window

  d_ymin = ymin;
  d_ymax = ymax;
  d_nextPoint = 0;
  d_axis_offset = 0.0;

  d_display_freq = DEFAULT_display_freq;
  d_ffts_per_display = DEFAULT_ffts_per_display;
  d_fft_count = 0;
}

template<class iType> void
GrFFTAvgSink<iType>::initialize()
{
  iType test_for_complex=0;
  double    samplingFrequency = getInputSamplingFrequencyN(0);	// Hz

  if (is_complex(test_for_complex)) {
    // this used to display the second half as "negative frequencies", but
    // it was screwed.  This just leaves them as is.  0 is the DC component.

    d_display = new VrGUIPlot (d_layout, FFT_XAXIS_NAME, d_label, true,
			       0, samplingFrequency,
			       d_ymin, d_ymax, d_nPoints, DIVISIONS);
    d_one_or_two = 1;
    for (int i = 0; i < d_nPoints; i++)
      d_xValues[i] = 
	((double)i / (double) d_nPoints * samplingFrequency) - d_axis_offset;
  }
  else {
    d_display = new VrGUIPlot (d_layout, FFT_XAXIS_NAME, d_label, true,
			       0, samplingFrequency/2,
			       d_ymin, d_ymax, d_nPoints/2, DIVISIONS);
    d_one_or_two = 2;
    for (int i = 0; i <= d_nPoints/2 ; i++)
      d_xValues[i] =
	((double)i / (double) d_nPoints * samplingFrequency) - d_axis_offset;
  }

  double fft_freq = d_display_freq * d_ffts_per_display;	// FFT's / second
  d_increment = (int) (samplingFrequency / fft_freq);
  if (d_increment < 1)
    d_increment = 1;

  set_skip_count ();

  std::cerr << "GrFFTAvgSink:\n"
	    << "  sampling_freq      = " << samplingFrequency << std::endl
	    << "  d_display_freq     = " << d_display_freq << std::endl
	    << "  d_ffts_per_display = " << d_ffts_per_display << std::endl
	    << "  fft_freq           = " << fft_freq << std::endl
	    << "  d_increment        = " << d_increment << std::endl
	    << "  d_skip_count       = " << d_skip_count << std::endl;
}

template<class iType> int
GrFFTAvgSink<iType>::work3(VrSampleRange output, 
			VrSampleRange inputs[], void *ai[])
{
  sync (output.index);

  collectData (((iType **) ai)[0], output.size);

  return output.size;
}

/*
 * Acquires the next point to plot on the graph.  Displays the graph when
 * all points are acquired.
 */
template<class iType> void
GrFFTAvgSink<iType>::collectData(iType *in, long samples_available)
{
  VrComplex	*fft_input = d_fft->get_inbuf ();
  VrComplex	*fft_output = d_fft->get_outbuf ();
  
  while (samples_available > 0){
    
    if (d_skip_count > 0){
      int	n = std::min ((long) d_skip_count, samples_available);
      d_skip_count -= n;

      in += n;
      samples_available -= n;
      continue;
    }

    if (d_nextPoint < d_nPoints)  {
      iType     	sampleValue;

      sampleValue = *in++;
      samples_available--;

      fft_input[d_nextPoint] = sampleValue * d_window[d_nextPoint];

      d_nextPoint++;
    }

    if (d_nextPoint >= d_nPoints)  {
      d_nextPoint = 0;
      set_skip_count ();
    
      d_fft->execute ();	// do the work

      // compute log magnitude
      //
      // this could be computed as 20 * log10 (abs (d_fft_output[i])), but
      // then you'd be wasting cycles computing the sqrt hidden in abs (z)
    
      for (int i = 0; i < d_nPoints ; i++){
	d_dbValues[i] =
	  10 * log10 (real(fft_output[i]) * real(fft_output[i])
		      + imag(fft_output[i]) * imag(fft_output[i]));

	d_avgdbValues[i] = (d_dbValues[i] * ALPHA
			    + d_avgdbValues[i] * (1.0 - ALPHA));
      }

      if (++d_fft_count >= d_ffts_per_display){
	d_display->data (d_xValues, d_dbValues, d_nPoints / d_one_or_two);
	d_fft_count = 0;
      }
    }
  }
}

template<class iType> 
GrFFTAvgSink<iType>::~GrFFTAvgSink()
{
  delete d_fft;
  delete[] d_xValues;
  delete[] d_dbValues;
  delete[] d_avgdbValues;
  delete[] d_window;
}
#endif
