/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRSKIPPINGSINK_H_
#define _VRSKIPPINGSINK_H_

#include <VrSink.h>
#include <stdio.h>
#include <assert.h>

//This sink uses a history-size chunk of data every increment units of data
//  (i.e. skips over increment-history units of data)
//  Each unit requested from the sink's work() procecure will operation
//  on a _block_ (history-size) of data.
template<class iType> 
class VrSkippingSink : public VrSink<iType> {
protected:
  unsigned int history;
  unsigned int increment;
public: 

  // FIXME history and increment should be arguments to constructor
  
  VrSkippingSink() : history(1), increment(1) {
    setOptimalSize(1);	/*forces size of 1 to allow skipping*/
  }

  virtual void pre_initialize () {
    if (increment <= 0) {
      fprintf (stderr, "[%s:%d] initialize() invalid value for increment=%d\n",
	       __FILE__, __LINE__, increment);
      abort ();
    }

    if (increment == 1){
      fprintf (stderr, "### Warning: increment == 1 in pre_initialize of %s.  Are you sure?\n",
	       name ());
    }

    setSamplingFrequency(getInputSamplingFrequencyN (0) / increment);
  }

  virtual int forecast(VrSampleRange output,
		       VrSampleRange inputs[]);

  virtual float averageInputUse(int n) {
    if(increment>history) return (float) history/(float) increment;
    else return 1.0;
  }

  virtual unsigned int mapSizeUp(int i, unsigned int size) {
    return (unsigned int) (size * averageInputUse(i) *
	   (getUpstreamModuleN(i)->getSamplingFrequency()
	    /getSamplingFrequency()));
  }

  virtual ~VrSkippingSink() {}
};

template<class iType> int
VrSkippingSink<iType>::forecast(VrSampleRange output,
				VrSampleRange inputs[]) {
  /* 1:history with increment skipping */
  /* size should always be 1 */
  assert (output.size == 1);

  for(unsigned int i=0;i<numberInputs;i++) {
    inputs[i].index=output.index*increment; /* ! do not subtract history ! */
    inputs[i].size=history;
  }
  return 0;
}  

#endif



















