/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VrHoppingComplexFIRfilter_H_
#define _VrHoppingComplexFIRfilter_H_

#include <VrDecimatingSigProc.h>

#if defined (ENABLE_MMX)
#include <VrMMX.h>
#endif

/*  FIR filter definition:

    VrHoppingComplexFIRfilter(N, Num of taps[], decimation factor, center frequency[], gain[])

    N = number of filters
    decimation factor => set to one unless some integer >1 specified
    cutoff (Hz) = 0.0  => LPF using Hamming window, o/w LPF transformed to have higher cutoff freq
    center_freq (Hz) => used to specify a composite frequency-shifting filter (i.e. channel filter)
    gain = the gain
*/

// ********************************************************

template<class iType> 
class VrHoppingComplexFIRfilter : public VrDecimatingSigProc<iType,VrComplex> {
protected:
  int *numTaps, num_ch;
  VrComplex** taps;
  VrComplex phase_correction;
  VrComplex *phase_corr_incr;
  long time;
  float *center_freq, *gain;
  int curr_ch;
  void buildFilter_complex(int);
#if defined (ENABLE_MMX)
  mmxTaps** processedTaps; //Precomputed constants, shifted four times
#endif
public: 
  /*
  virtual void skip(int n) { 
    if(num_ch>1) {
      //until skip works right with multiple outputBuffers, we just call work
      work(n);
    } else
      VrDecimatingSigProc<iType,VrComplex>::skip(n);
  }
  */
  virtual void work(int n);
  virtual void initialize();
  int setCenter_Freq(int, float);
  int setCenter_Freq(float);
  int setNumber_Taps(int,int);
  int setNumber_Taps(int);
  void changeChannel(int);
  VrHoppingComplexFIRfilter(int n, int d, const int t[], const float f[], 
		     const float g[]);
  VrHoppingComplexFIRfilter(int d, int t,float f, float g);
  ~VrHoppingComplexFIRfilter();
};

template<class iType> void
VrHoppingComplexFIRfilter<iType>::work(int n)
{
  VrComplex result = 0;

  for (int i=0;i<n;i++,incReadPtr(decimation)) {
    result = 0;

    //make input pointer local
    iType *inputArray = inputReadPtr(-numTaps[curr_ch]+1);
    
#if defined (ENABLE_MMX)
    if(processedTaps[curr_ch]->mmxReady())
      result = processedTaps[curr_ch]->mmxCVDProduct(inputArray);
    else { 
      VrComplex *taps_tmp = taps[curr_ch];
      for (int j=0; j < numTaps[curr_ch]; j++)
	result += taps_tmp[j] * inputArray[j];
    }
#else
    VrComplex *taps_tmp = taps[curr_ch];
    for (int j=0; j < numTaps[curr_ch]; j++)
      result += taps_tmp[j] * inputArray[j];     
#endif
    
    // Perform phase correction (non-trivial only for freq-xlating filter)
    if (center_freq[curr_ch] != 0.0) {
      phase_correction *= phase_corr_incr[curr_ch];
      result *= phase_correction;
    }
    outputWrite(result);
  }
  
}

template<class iType> void
VrHoppingComplexFIRfilter<iType>::buildFilter_complex(int ch){
  int inSampFreq;
  int index;
  float N = numTaps[ch];
  float M = N-1; /* filter Order */

  inSampFreq = getInputSamplingFrequencyN(0); 
  time = 0;
  
  if (center_freq[ch] == 0.0){

      // Build Complex Filter => 
      //            produces a low-pass filter using a real Hamming window

      for ( index=0 ; index < numTaps[ch] ; index++) {
       taps[ch][index] = gain[ch]*VrComplex((0.54-0.46*cos(2*M_PI*index/(M))));
    }
    
  } else {

    // Build composite Complex Filter => adds freq-shifting part

    float arg = 2*M_PI*center_freq[ch] / (float)inSampFreq;
    for ( index=0 ; index < numTaps[ch] ; index++) {

      taps[ch][index] = VrComplex(gain[ch]*cos(arg*index)*(0.54-0.46*cos(2*M_PI*index/(M))),
         gain[ch]*(-1)*sin(arg*index)*(0.54-0.46*cos(2*M_PI*index/(M))));

    }
    phase_corr_incr[ch] = VrComplex(cos(arg*(float)decimation),
				(-1)*sin(arg*(float)decimation));
  }


#if defined (ENABLE_MMX)
  if(processedTaps[ch]!=NULL)
    delete processedTaps[ch];
  processedTaps[ch]=new mmxTaps(taps[ch],numTaps[ch]);
#endif

}


template<class iType> 
VrHoppingComplexFIRfilter<iType>::VrHoppingComplexFIRfilter(int n, int dec, const int t[], 
					      const float freq[], 
					      const float g[])
  :VrDecimatingSigProc<iType,VrComplex>(n,dec), num_ch(n),curr_ch(0)
{

  numTaps=new int[num_ch];
  phase_corr_incr = new VrComplex[num_ch];
  center_freq = new float[num_ch];
  gain = new float[num_ch];
  taps = new (VrComplex *[num_ch]);

#if defined (ENABLE_MMX)
  processedTaps = new (mmxTaps *[num_ch]);
#endif

  phase_correction = VrComplex(1,0);  
  for (int i=0; i<num_ch; i++){
    numTaps[i] = t[i];
    phase_corr_incr[i] = VrComplex(1,0);
    center_freq[i] = freq[i];
    gain[i] = g[i];
#if defined (ENABLE_MMX)
  processedTaps[i]=NULL;
#endif
  }

}

template<class iType> 
VrHoppingComplexFIRfilter<iType>::VrHoppingComplexFIRfilter(int dec,int t,float freq, float g)
  :VrDecimatingSigProc<iType,VrComplex>(1,dec), num_ch(1),curr_ch(0)
{

  numTaps=new int[num_ch];
  phase_corr_incr = new VrComplex[num_ch];
  center_freq = new float[num_ch];
  gain = new float[num_ch];
  taps = new (VrComplex *[num_ch]);

#if defined (ENABLE_MMX)
  processedTaps = new (mmxTaps *[num_ch]);
#endif
  
  numTaps[0] = t;
  phase_correction = VrComplex(1,0);
  phase_corr_incr[0] = VrComplex(1,0);
  center_freq[0] = freq;
  gain[0] = g;
#if defined (ENABLE_MMX)
  processedTaps[0]=NULL;
#endif
}

template<class iType> 
void VrHoppingComplexFIRfilter<iType>::initialize()
{
  for (int i=0; i<num_ch; i++){ 
    taps[i]=new VrComplex[numTaps[i]];
    buildFilter_complex(i);
  }

  //Set history
  int max_numTaps = 0;
  for (int i=0; i<num_ch; i++){
    if (numTaps[i] > max_numTaps) max_numTaps = numTaps[i];
  }
  setHistory(max_numTaps);
}

template<class iType> 
int VrHoppingComplexFIRfilter<iType>::setCenter_Freq(int ch, float cf)
{
  center_freq[ch] = cf;
  buildFilter_complex(ch);
  return 1;
}

template<class iType> 
int VrHoppingComplexFIRfilter<iType>::setCenter_Freq(float cf)
{
  return setCenter_Freq(0,cf);
}

template<class iType> 
int VrHoppingComplexFIRfilter<iType>::setNumber_Taps(int ch, int numT)
{
  numTaps[ch] = numT;
  delete taps[ch];
  taps[ch]=new VrComplex[numTaps[ch]];

  //set history
  int max_numTaps = 0;
  for (int i=0; i<num_ch; i++){
    if (numTaps[i] > max_numTaps) max_numTaps = numTaps[i];
  }
  setHistory(max_numTaps);

  buildFilter_complex(ch);
  return 1;
}

template<class iType> 
int VrHoppingComplexFIRfilter<iType>::setNumber_Taps(int numT)
{
  return setNumber_Taps(0, numT);
}

template<class iType>
void VrHoppingComplexFIRfilter<iType>::changeChannel(int ch)
{
  curr_ch = ch;
  return;
}

template<class iType> 
VrHoppingComplexFIRfilter<iType>::~VrHoppingComplexFIRfilter()
{
  
  for (int i=0; i<num_ch; i++){ 
    delete taps[i];
#if defined (ENABLE_MMX)
    if(processedTaps[i]!=NULL)
      delete processedTaps[i];
#endif
  }
  delete numTaps;
  delete [] phase_corr_incr;
  delete center_freq;
  delete gain;
  delete taps;

}



#endif



















